'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _metaApiConnection = require('./metaApiConnection');

var _metaApiConnection2 = _interopRequireDefault(_metaApiConnection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Exposes MetaApi MetaTrader RPC API connection to consumers
 */
class RpcMetaApiConnection extends _metaApiConnection2.default {

  /**
   * Constructs MetaApi MetaTrader RPC Api connection
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   */
  constructor(websocketClient, account) {
    super(websocketClient, account, 'RPC');
    this._logger = _logger2.default.getLogger('MetaApiConnection');
  }

  /**
   * Returns account information (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readAccountInformation/).
   * @returns {Promise<MetatraderAccountInformation>} promise resolving with account information
   */
  getAccountInformation() {
    return this._websocketClient.getAccountInformation(this._account.id);
  }

  /**
   * Returns positions (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readPositions/).
   * @returns {Promise<Array<MetatraderPosition>} promise resolving with array of open positions
   */
  getPositions() {
    return this._websocketClient.getPositions(this._account.id);
  }

  /**
   * Returns specific position (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readPosition/).
   * @param {String} positionId position id
   * @return {Promise<MetatraderPosition>} promise resolving with MetaTrader position found
   */
  getPosition(positionId) {
    return this._websocketClient.getPosition(this._account.id, positionId);
  }

  /**
   * Returns open orders (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readOrders/).
   * @return {Promise<Array<MetatraderOrder>>} promise resolving with open MetaTrader orders
   */
  getOrders() {
    return this._websocketClient.getOrders(this._account.id);
  }

  /**
   * Returns specific open order (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readOrder/).
   * @param {String} orderId order id (ticket number)
   * @return {Promise<MetatraderOrder>} promise resolving with metatrader order found
   */
  getOrder(orderId) {
    return this._websocketClient.getOrder(this._account.id, orderId);
  }

  /**
   * Returns the history of completed orders for a specific ticket number (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readHistoryOrdersByTicket/).
   * @param {String} ticket ticket number (order id)
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  getHistoryOrdersByTicket(ticket) {
    return this._websocketClient.getHistoryOrdersByTicket(this._account.id, ticket);
  }

  /**
   * Returns the history of completed orders for a specific position id (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readHistoryOrdersByPosition/)
   * @param {String} positionId position id
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  getHistoryOrdersByPosition(positionId) {
    return this._websocketClient.getHistoryOrdersByPosition(this._account.id, positionId);
  }

  /**
   * Returns the history of completed orders for a specific time range (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readHistoryOrdersByTimeRange/)
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  getHistoryOrdersByTimeRange(startTime, endTime, offset = 0, limit = 1000) {
    return this._websocketClient.getHistoryOrdersByTimeRange(this._account.id, startTime, endTime, offset, limit);
  }

  /**
   * Returns history deals with a specific ticket number (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readDealsByTicket/).
   * @param {String} ticket ticket number (deal id for MT5 or order id for MT4)
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  getDealsByTicket(ticket) {
    return this._websocketClient.getDealsByTicket(this._account.id, ticket);
  }

  /**
   * Returns history deals for a specific position id (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readDealsByPosition/).
   * @param {String} positionId position id
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  getDealsByPosition(positionId) {
    return this._websocketClient.getDealsByPosition(this._account.id, positionId);
  }

  /**
   * Returns history deals with for a specific time range (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readDealsByTimeRange/).
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  getDealsByTimeRange(startTime, endTime, offset = 0, limit = 1000) {
    return this._websocketClient.getDealsByTimeRange(this._account.id, startTime, endTime, offset, limit);
  }

  /**
   * Clears the order and transaction history of a specified application so that it can be synchronized from scratch
   * (see https://metaapi.cloud/docs/client/websocket/api/removeHistory/).
   * @param {String} [application] application to remove history for
   * @return {Promise} promise resolving when the history is cleared
   */
  removeHistory(application) {
    return this._websocketClient.removeHistory(this._account.id, application);
  }

  /**
   * Retrieves available symbols for an account (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readSymbols/).
   * @param {String} symbol symbol to retrieve symbols for
   * @returns {Promise<Array<string>>} promise which resolves when symbols are retrieved
   */
  getSymbols() {
    return this._websocketClient.getSymbols(this._account.id);
  }

  /**
   * Retrieves specification for a symbol (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readSymbolSpecification/).
   * @param {String} symbol symbol to retrieve specification for
   * @returns {Promise<MetatraderSymbolSpecification>} promise which resolves when specification is retrieved
   */
  getSymbolSpecification(symbol) {
    return this._websocketClient.getSymbolSpecification(this._account.id, symbol);
  }

  /**
   * Retrieves latest price for a symbol (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readSymbolPrice/).
   * @param {String} symbol symbol to retrieve price for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderSymbolPrice>} promise which resolves when price is retrieved
   */
  getSymbolPrice(symbol, keepSubscription) {
    return this._websocketClient.getSymbolPrice(this._account.id, symbol, keepSubscription);
  }

  /**
   * Retrieves latest candle for a symbol and timeframe (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readCandle/).
   * @param {String} symbol symbol to retrieve candle for
   * @param {string} timeframe defines the timeframe according to which the candle must be generated. Allowed values for
   * MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values
   * for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderCandle>} promise which resolves when candle is retrieved
   */
  getCandle(symbol, timeframe, keepSubscription = false) {
    return this._websocketClient.getCandle(this._account.id, symbol, timeframe, keepSubscription);
  }

  /**
   * Retrieves latest tick for a symbol. MT4 G1 accounts do not support this API (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readTick/).
   * @param {String} symbol symbol to retrieve tick for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when tick is retrieved
   */
  getTick(symbol, keepSubscription = false) {
    return this._websocketClient.getTick(this._account.id, symbol, keepSubscription);
  }

  /**
   * Retrieves latest order book for a symbol. MT4 accounts do not support this API (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readBook/).
   * @param {string} symbol symbol to retrieve order book for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when order book is retrieved
   */
  getBook(symbol, keepSubscription = false) {
    return this._websocketClient.getBook(this._account.id, symbol, keepSubscription);
  }

  /**
   * Waits until synchronization to RPC application is completed
   * @param {Number} timeoutInSeconds synchronization timeout in seconds
   * @return {Promise} promise which resolves when synchronization to RPC application is completed
   * @throws {TimeoutError} if application failed to synchronize with the teminal withing timeout allowed
   */
  async waitSynchronized(timeoutInSeconds = 300) {
    const startTime = Date.now();
    // eslint-disable-next-line
    while (true) {
      try {
        await this._websocketClient.waitSynchronized(this._account.id, 0, 'RPC', 5, 'RPC');
        break;
      } catch (err) {
        if (Date.now() > startTime + timeoutInSeconds * 1000) {
          throw err;
        }
      }
    }
  }

}
exports.default = RpcMetaApiConnection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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