'use strict';

/**
 * Implements a provisioning profile entity
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class ProvisioningProfile {

  /**
   * Constructs a provisioning profile entity
   * @param {ProvisioningProfileDto} data provisioning profile data
   * @param {ProvisioningProfileClient} provisioningProfileClient provisioning profile REST API client
   */
  constructor(data, provisioningProfileClient) {
    this._data = data;
    this._provisioningProfileClient = provisioningProfileClient;
  }

  /**
   * Returns profile id
   * @return {String} profile id
   */
  get id() {
    return this._data._id;
  }

  /**
   * Returns profile name
   * @return {String} profile name
   */
  get name() {
    return this._data.name;
  }

  /**
   * Returns profile version. Possible values are 4 and 5
   * @return {Number} profile version
   */
  get version() {
    return this._data.version;
  }

  /**
   * Returns profile status. Possible values are new and active
   * @return {String} profile status
   */
  get status() {
    return this._data.status;
  }

  /**
   * Returns broker timezone name from Time Zone Database
   * @return {String} broker timezone name
   */
  get brokerTimezone() {
    return this._data.brokerTimezone;
  }

  /**
   * Returns broker DST timezone name from Time Zone Database
   * @return {String} broker DST switch timezone name
   */
  get brokerDSTSwitchTimezone() {
    return this._data.brokerDSTSwitchTimezone;
  }

  /**
   * Reloads provisioning profile from API
   * @return {Promise} promise resolving when provisioning profile is updated
   */
  async reload() {
    this._data = await this._provisioningProfileClient.getProvisioningProfile(this.id);
  }

  /**
   * Removes provisioning profile. The current object instance should be discarded after returned promise resolves.
   * @return {Promise} promise resolving when provisioning profile is removed
   */
  remove() {
    return this._provisioningProfileClient.deleteProvisioningProfile(this.id);
  }

  /**
   * Uploads a file to provisioning profile.
   * @param {String} fileName name of the file to upload. Allowed values are servers.dat for MT5 profile, broker.srv for
   * MT4 profile
   * @param {String|Buffer} file path to a file to upload or buffer containing file contents
   * @return {Promise} promise which resolves when the file was uploaded
   */
  uploadFile(fileName, file) {
    return this._provisioningProfileClient.uploadProvisioningProfileFile(this.id, fileName, file);
  }

  /**
   * Updates provisioning profile
   * @param {ProvisioningProfileUpdateDto} profile provisioning profile update
   * @return {Promise} promise resolving when provisioning profile is updated
   */
  async update(profile) {
    await this._provisioningProfileClient.updateProvisioningProfile(this.id, profile);
    await this.reload();
  }

}
exports.default = ProvisioningProfile;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL2xpYi9tZXRhQXBpL3Byb3Zpc2lvbmluZ1Byb2ZpbGUuZXM2Il0sIm5hbWVzIjpbIlByb3Zpc2lvbmluZ1Byb2ZpbGUiLCJjb25zdHJ1Y3RvciIsImRhdGEiLCJwcm92aXNpb25pbmdQcm9maWxlQ2xpZW50IiwiX2RhdGEiLCJfcHJvdmlzaW9uaW5nUHJvZmlsZUNsaWVudCIsImlkIiwiX2lkIiwibmFtZSIsInZlcnNpb24iLCJzdGF0dXMiLCJicm9rZXJUaW1lem9uZSIsImJyb2tlckRTVFN3aXRjaFRpbWV6b25lIiwicmVsb2FkIiwiZ2V0UHJvdmlzaW9uaW5nUHJvZmlsZSIsInJlbW92ZSIsImRlbGV0ZVByb3Zpc2lvbmluZ1Byb2ZpbGUiLCJ1cGxvYWRGaWxlIiwiZmlsZU5hbWUiLCJmaWxlIiwidXBsb2FkUHJvdmlzaW9uaW5nUHJvZmlsZUZpbGUiLCJ1cGRhdGUiLCJwcm9maWxlIiwidXBkYXRlUHJvdmlzaW9uaW5nUHJvZmlsZSJdLCJtYXBwaW5ncyI6IkFBQUE7O0FBRUE7Ozs7Ozs7QUFHZSxNQUFNQSxtQkFBTixDQUEwQjs7QUFFdkM7Ozs7O0FBS0FDLGNBQVlDLElBQVosRUFBa0JDLHlCQUFsQixFQUE2QztBQUMzQyxTQUFLQyxLQUFMLEdBQWFGLElBQWI7QUFDQSxTQUFLRywwQkFBTCxHQUFrQ0YseUJBQWxDO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJRyxFQUFKLEdBQVM7QUFDUCxXQUFPLEtBQUtGLEtBQUwsQ0FBV0csR0FBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLElBQUosR0FBVztBQUNULFdBQU8sS0FBS0osS0FBTCxDQUFXSSxJQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsT0FBSixHQUFjO0FBQ1osV0FBTyxLQUFLTCxLQUFMLENBQVdLLE9BQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJQyxNQUFKLEdBQWE7QUFDWCxXQUFPLEtBQUtOLEtBQUwsQ0FBV00sTUFBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLGNBQUosR0FBcUI7QUFDbkIsV0FBTyxLQUFLUCxLQUFMLENBQVdPLGNBQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJQyx1QkFBSixHQUE4QjtBQUM1QixXQUFPLEtBQUtSLEtBQUwsQ0FBV1EsdUJBQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxRQUFNQyxNQUFOLEdBQWU7QUFDYixTQUFLVCxLQUFMLEdBQWEsTUFBTSxLQUFLQywwQkFBTCxDQUFnQ1Msc0JBQWhDLENBQXVELEtBQUtSLEVBQTVELENBQW5CO0FBQ0Q7O0FBRUQ7Ozs7QUFJQVMsV0FBUztBQUNQLFdBQU8sS0FBS1YsMEJBQUwsQ0FBZ0NXLHlCQUFoQyxDQUEwRCxLQUFLVixFQUEvRCxDQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7QUFPQVcsYUFBV0MsUUFBWCxFQUFxQkMsSUFBckIsRUFBMkI7QUFDekIsV0FBTyxLQUFLZCwwQkFBTCxDQUFnQ2UsNkJBQWhDLENBQThELEtBQUtkLEVBQW5FLEVBQXVFWSxRQUF2RSxFQUFpRkMsSUFBakYsQ0FBUDtBQUNEOztBQUVEOzs7OztBQUtBLFFBQU1FLE1BQU4sQ0FBYUMsT0FBYixFQUFzQjtBQUNwQixVQUFNLEtBQUtqQiwwQkFBTCxDQUFnQ2tCLHlCQUFoQyxDQUEwRCxLQUFLakIsRUFBL0QsRUFBbUVnQixPQUFuRSxDQUFOO0FBQ0EsVUFBTSxLQUFLVCxNQUFMLEVBQU47QUFDRDs7QUEvRnNDO2tCQUFwQmIsbUIiLCJmaWxlIjoicHJvdmlzaW9uaW5nUHJvZmlsZS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxuLyoqXG4gKiBJbXBsZW1lbnRzIGEgcHJvdmlzaW9uaW5nIHByb2ZpbGUgZW50aXR5XG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIFByb3Zpc2lvbmluZ1Byb2ZpbGUge1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgcHJvdmlzaW9uaW5nIHByb2ZpbGUgZW50aXR5XG4gICAqIEBwYXJhbSB7UHJvdmlzaW9uaW5nUHJvZmlsZUR0b30gZGF0YSBwcm92aXNpb25pbmcgcHJvZmlsZSBkYXRhXG4gICAqIEBwYXJhbSB7UHJvdmlzaW9uaW5nUHJvZmlsZUNsaWVudH0gcHJvdmlzaW9uaW5nUHJvZmlsZUNsaWVudCBwcm92aXNpb25pbmcgcHJvZmlsZSBSRVNUIEFQSSBjbGllbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yKGRhdGEsIHByb3Zpc2lvbmluZ1Byb2ZpbGVDbGllbnQpIHtcbiAgICB0aGlzLl9kYXRhID0gZGF0YTtcbiAgICB0aGlzLl9wcm92aXNpb25pbmdQcm9maWxlQ2xpZW50ID0gcHJvdmlzaW9uaW5nUHJvZmlsZUNsaWVudDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHByb2ZpbGUgaWRcbiAgICogQHJldHVybiB7U3RyaW5nfSBwcm9maWxlIGlkXG4gICAqL1xuICBnZXQgaWQoKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RhdGEuX2lkO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgcHJvZmlsZSBuYW1lXG4gICAqIEByZXR1cm4ge1N0cmluZ30gcHJvZmlsZSBuYW1lXG4gICAqL1xuICBnZXQgbmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5uYW1lO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgcHJvZmlsZSB2ZXJzaW9uLiBQb3NzaWJsZSB2YWx1ZXMgYXJlIDQgYW5kIDVcbiAgICogQHJldHVybiB7TnVtYmVyfSBwcm9maWxlIHZlcnNpb25cbiAgICovXG4gIGdldCB2ZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLnZlcnNpb247XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBwcm9maWxlIHN0YXR1cy4gUG9zc2libGUgdmFsdWVzIGFyZSBuZXcgYW5kIGFjdGl2ZVxuICAgKiBAcmV0dXJuIHtTdHJpbmd9IHByb2ZpbGUgc3RhdHVzXG4gICAqL1xuICBnZXQgc3RhdHVzKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLnN0YXR1cztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGJyb2tlciB0aW1lem9uZSBuYW1lIGZyb20gVGltZSBab25lIERhdGFiYXNlXG4gICAqIEByZXR1cm4ge1N0cmluZ30gYnJva2VyIHRpbWV6b25lIG5hbWVcbiAgICovXG4gIGdldCBicm9rZXJUaW1lem9uZSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5icm9rZXJUaW1lem9uZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGJyb2tlciBEU1QgdGltZXpvbmUgbmFtZSBmcm9tIFRpbWUgWm9uZSBEYXRhYmFzZVxuICAgKiBAcmV0dXJuIHtTdHJpbmd9IGJyb2tlciBEU1Qgc3dpdGNoIHRpbWV6b25lIG5hbWVcbiAgICovXG4gIGdldCBicm9rZXJEU1RTd2l0Y2hUaW1lem9uZSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5icm9rZXJEU1RTd2l0Y2hUaW1lem9uZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWxvYWRzIHByb3Zpc2lvbmluZyBwcm9maWxlIGZyb20gQVBJXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gcHJvdmlzaW9uaW5nIHByb2ZpbGUgaXMgdXBkYXRlZFxuICAgKi9cbiAgYXN5bmMgcmVsb2FkKCkge1xuICAgIHRoaXMuX2RhdGEgPSBhd2FpdCB0aGlzLl9wcm92aXNpb25pbmdQcm9maWxlQ2xpZW50LmdldFByb3Zpc2lvbmluZ1Byb2ZpbGUodGhpcy5pZCk7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlcyBwcm92aXNpb25pbmcgcHJvZmlsZS4gVGhlIGN1cnJlbnQgb2JqZWN0IGluc3RhbmNlIHNob3VsZCBiZSBkaXNjYXJkZWQgYWZ0ZXIgcmV0dXJuZWQgcHJvbWlzZSByZXNvbHZlcy5cbiAgICogQHJldHVybiB7UHJvbWlzZX0gcHJvbWlzZSByZXNvbHZpbmcgd2hlbiBwcm92aXNpb25pbmcgcHJvZmlsZSBpcyByZW1vdmVkXG4gICAqL1xuICByZW1vdmUoKSB7XG4gICAgcmV0dXJuIHRoaXMuX3Byb3Zpc2lvbmluZ1Byb2ZpbGVDbGllbnQuZGVsZXRlUHJvdmlzaW9uaW5nUHJvZmlsZSh0aGlzLmlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGxvYWRzIGEgZmlsZSB0byBwcm92aXNpb25pbmcgcHJvZmlsZS5cbiAgICogQHBhcmFtIHtTdHJpbmd9IGZpbGVOYW1lIG5hbWUgb2YgdGhlIGZpbGUgdG8gdXBsb2FkLiBBbGxvd2VkIHZhbHVlcyBhcmUgc2VydmVycy5kYXQgZm9yIE1UNSBwcm9maWxlLCBicm9rZXIuc3J2IGZvclxuICAgKiBNVDQgcHJvZmlsZVxuICAgKiBAcGFyYW0ge1N0cmluZ3xCdWZmZXJ9IGZpbGUgcGF0aCB0byBhIGZpbGUgdG8gdXBsb2FkIG9yIGJ1ZmZlciBjb250YWluaW5nIGZpbGUgY29udGVudHNcbiAgICogQHJldHVybiB7UHJvbWlzZX0gcHJvbWlzZSB3aGljaCByZXNvbHZlcyB3aGVuIHRoZSBmaWxlIHdhcyB1cGxvYWRlZFxuICAgKi9cbiAgdXBsb2FkRmlsZShmaWxlTmFtZSwgZmlsZSkge1xuICAgIHJldHVybiB0aGlzLl9wcm92aXNpb25pbmdQcm9maWxlQ2xpZW50LnVwbG9hZFByb3Zpc2lvbmluZ1Byb2ZpbGVGaWxlKHRoaXMuaWQsIGZpbGVOYW1lLCBmaWxlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHByb3Zpc2lvbmluZyBwcm9maWxlXG4gICAqIEBwYXJhbSB7UHJvdmlzaW9uaW5nUHJvZmlsZVVwZGF0ZUR0b30gcHJvZmlsZSBwcm92aXNpb25pbmcgcHJvZmlsZSB1cGRhdGVcbiAgICogQHJldHVybiB7UHJvbWlzZX0gcHJvbWlzZSByZXNvbHZpbmcgd2hlbiBwcm92aXNpb25pbmcgcHJvZmlsZSBpcyB1cGRhdGVkXG4gICAqL1xuICBhc3luYyB1cGRhdGUocHJvZmlsZSkge1xuICAgIGF3YWl0IHRoaXMuX3Byb3Zpc2lvbmluZ1Byb2ZpbGVDbGllbnQudXBkYXRlUHJvdmlzaW9uaW5nUHJvZmlsZSh0aGlzLmlkLCBwcm9maWxlKTtcbiAgICBhd2FpdCB0aGlzLnJlbG9hZCgpO1xuICB9XG5cbn1cbiJdfQ==