'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _timeoutError = require('../clients/timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _rpcMetaApiConnection = require('./rpcMetaApiConnection');

var _rpcMetaApiConnection2 = _interopRequireDefault(_rpcMetaApiConnection);

var _index = require('./historyFileManager/index');

var _index2 = _interopRequireDefault(_index);

var _expertAdvisor = require('./expertAdvisor');

var _expertAdvisor2 = _interopRequireDefault(_expertAdvisor);

var _errorHandler = require('../clients/errorHandler');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Implements a MetaTrader account entity
 */
class MetatraderAccount {

  /**
   * Constructs a MetaTrader account entity
   * @param {MetatraderAccountDto} data MetaTrader account data
   * @param {MetatraderAccountClient} metatraderAccountClient MetaTrader account REST API client
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {ExpertAdvisorClient} expertAdvisorClient expert advisor REST API client
   * @param {HistoricalMarketDataClient} historicalMarketDataClient historical market data HTTP API client
   */
  constructor(data, metatraderAccountClient, metaApiWebsocketClient, connectionRegistry, expertAdvisorClient, historicalMarketDataClient) {
    this._data = data;
    this._metatraderAccountClient = metatraderAccountClient;
    this._metaApiWebsocketClient = metaApiWebsocketClient;
    this._connectionRegistry = connectionRegistry;
    this._expertAdvisorClient = expertAdvisorClient;
    this._historicalMarketDataClient = historicalMarketDataClient;
  }

  /**
   * Returns account id
   * @return {String} account id
   */
  get id() {
    return this._data._id;
  }

  /**
   * Returns account name
   * @return {String} account name
   */
  get name() {
    return this._data.name;
  }

  /**
   * Returns account type. Possible values are cloud, cloud-g1, cloud-g2 and self-hosted.
   * @return {String} account type
   */
  get type() {
    return this._data.type;
  }

  /**
   * Returns account login
   * @return {String} account login
   */
  get login() {
    return this._data.login;
  }

  /**
   * Returns MetaTrader server which hosts the account
   * @return {String} MetaTrader server which hosts the account
   */
  get server() {
    return this._data.server;
  }

  /**
   * Returns id of the account's provisioning profile
   * @return {String} id of the account's provisioning profile
   */
  get provisioningProfileId() {
    return this._data.provisioningProfileId;
  }

  /**
   * Returns application name to connect the account to. Currently allowed values are MetaApi and AgiliumTrade
   * @return {String} application name to connect the account to
   */
  get application() {
    return this._data.application;
  }

  /**
   * Returns MetaTrader magic to place trades using
   * @return {Number} MetaTrader magic to place trades using
   */
  get magic() {
    return this._data.magic;
  }

  /**
   * Returns account deployment state. One of CREATED, DEPLOYING, DEPLOYED, UNDEPLOYING, UNDEPLOYED, DELETING
   * @return {String} account deployment state
   */
  get state() {
    return this._data.state;
  }

  /**
   * Returns terminal & broker connection status, one of CONNECTED, DISCONNECTED, DISCONNECTED_FROM_BROKER
   * @return {String} terminal & broker connection status
   */
  get connectionStatus() {
    return this._data.connectionStatus;
  }

  /**
   * Returns authorization access token to be used for accessing single account data.
   * Intended to be used in browser API.
   * @return {String} authorization token
   */
  get accessToken() {
    return this._data.accessToken;
  }

  /**
   * Returns flag indicating if trades should be placed as manual trades on this account
   * @return {Boolean} flag indicating if trades should be placed as manual trades on this account
   */
  get manualTrades() {
    return !!this._data.manualTrades;
  }

  /**
   * Returns API extensions
   * @return {Array<Extension>} API extensions
   */
  get extensions() {
    return this._data.extensions;
  }

  /**
   * Returns extra information which can be stored together with your account
   * @return {Object} extra information which can be stored together with your account
   */
  get metadata() {
    return this._data.metadata;
  }

  /**
   * Returns user-defined account tags
   * @return {Array<string>} user-defined account tags
   */
  get tags() {
    return this._data.tags;
  }

  /**
   * Returns account roles for CopyFactory2 application
   * @return {Array<string>} account roles for CopyFactory2 application
   */
  get copyFactoryRoles() {
    return this._data.copyFactoryRoles;
  }

  /**
   * Returns number of resource slots to allocate to account. Allocating extra resource slots
   * results in better account performance under load which is useful for some applications. E.g. if you have many
   * accounts copying the same strategy via CooyFactory API, then you can increase resourceSlots to get a lower trade
   * copying latency. Please note that allocating extra resource slots is a paid option. Please note that high
   * reliability accounts use redundant infrastructure, so that each resource slot for a high reliability account
   * is billed as 2 standard resource slots.  Default is 1.
   * @return {number} number of resource slots to allocate to account
   */
  get resourceSlots() {
    return this._data.resourceSlots;
  }

  /**
   * Returns the number of CopyFactory 2 resource slots to allocate to account.
   * Allocating extra resource slots results in lower trade copying latency. Please note that allocating extra resource
   * slots is a paid option. Please also note that CopyFactory 2 uses redundant infrastructure so that
   * each CopyFactory resource slot is billed as 2 standard resource slots. You will be billed for CopyFactory 2
   * resource slots only if you have added your account to CopyFactory 2 by specifying copyFactoryRoles field.
   * Default is 1.
   * @return {number} number of CopyFactory 2 resource slots to allocate to account
   */
  get copyFactoryResourceSlots() {
    return this._data.copyFactoryResourceSlots;
  }

  /**
   * Returns 3-character ISO currency code of the account base currency. Default value is USD. The setting is to be used
   * for copy trading accounts which use national currencies only, such as some Brazilian brokers. You should not alter
   * this setting unless you understand what you are doing.
   * @return {number} 3-character ISO currency code of the account base currency
   */
  get baseCurrency() {
    return this._data.baseCurrency;
  }

  /**
   * Returns reliability value. Possible values are regular and high
   * @return {String} account reliability value
   */
  get reliability() {
    return this._data.reliability;
  }

  /**
   * Returns version value. Possible values are 4 and 5
   * @return {String} account version value
   */
  get version() {
    return this._data.version;
  }

  /**
   * Reloads MetaTrader account from API
   * @return {Promise} promise resolving when MetaTrader account is updated
   */
  async reload() {
    this._data = await this._metatraderAccountClient.getAccount(this.id);
  }

  /**
   * Removes MetaTrader account. Cloud account transitions to DELETING state. 
   * It takes some time for an account to be eventually deleted. Self-hosted 
   * account is deleted immediately.
   * @return {Promise} promise resolving when account is scheduled for deletion
   */
  async remove() {
    this._connectionRegistry.remove(this.id);
    await this._metatraderAccountClient.deleteAccount(this.id);
    const fileManager = new _index2.default(this.id);
    await fileManager.deleteStorageFromDisk();
    if (this.type !== 'self-hosted') {
      try {
        await this.reload();
      } catch (err) {
        if (err.name !== 'NotFoundError') {
          throw err;
        }
      }
    }
  }

  /**
   * Schedules account for deployment. It takes some time for API server to be started and account to reach the DEPLOYED
   * state
   * @returns {Promise} promise resolving when account is scheduled for deployment
   */
  async deploy() {
    await this._metatraderAccountClient.deployAccount(this.id);
    await this.reload();
  }

  /**
   * Schedules account for undeployment. It takes some time for API server to be stopped and account to reach the
   * UNDEPLOYED state
   * @returns {Promise} promise resolving when account is scheduled for undeployment
   */
  async undeploy() {
    this._connectionRegistry.remove(this.id);
    await this._metatraderAccountClient.undeployAccount(this.id);
    await this.reload();
  }

  /**
   * Schedules account for redeployment. It takes some time for API server to be restarted and account to reach the
   * DEPLOYED state
   * @returns {Promise} promise resolving when account is scheduled for redeployment
   */
  async redeploy() {
    await this._metatraderAccountClient.redeployAccount(this.id);
    await this.reload();
  }

  /**
   * Increases MetaTrader account reliability. The account will be temporary stopped to perform this action
   * @returns {Promise} promise resolving when account reliability is increased
   */
  async increaseReliability() {
    await this._metatraderAccountClient.increaseReliability(this.id);
    await this.reload();
  }

  /**
   * Waits until API server has finished deployment and account reached the DEPLOYED state
   * @param {Number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {Number} intervalInMilliseconds interval between account reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when account is deployed
   * @throws {TimeoutError} if account have not reached the DEPLOYED state withing timeout allowed
   */
  async waitDeployed(timeoutInSeconds = 300, intervalInMilliseconds = 1000) {
    let startTime = Date.now();
    await this.reload();
    while (this.state !== 'DEPLOYED' && startTime + timeoutInSeconds * 1000 > Date.now()) {
      await this._delay(intervalInMilliseconds);
      await this.reload();
    }
    if (this.state !== 'DEPLOYED') {
      throw new _timeoutError2.default('Timed out waiting for account ' + this.id + ' to be deployed');
    }
  }

  /**
   * Waits until API server has finished undeployment and account reached the UNDEPLOYED state
   * @param {Number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {Number} intervalInMilliseconds interval between account reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when account is deployed
   * @throws {TimeoutError} if account have not reached the UNDEPLOYED state withing timeout allowed
   */
  async waitUndeployed(timeoutInSeconds = 300, intervalInMilliseconds = 1000) {
    let startTime = Date.now();
    await this.reload();
    while (this.state !== 'UNDEPLOYED' && startTime + timeoutInSeconds * 1000 > Date.now()) {
      await this._delay(intervalInMilliseconds);
      await this.reload();
    }
    if (this.state !== 'UNDEPLOYED') {
      throw new _timeoutError2.default('Timed out waiting for account ' + this.id + ' to be undeployed');
    }
  }

  /**
   * Waits until account has been deleted
   * @param {Number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {Number} intervalInMilliseconds interval between account reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when account is deleted
   * @throws {TimeoutError} if account was not deleted withing timeout allowed
   */
  async waitRemoved(timeoutInSeconds = 300, intervalInMilliseconds = 1000) {
    let startTime = Date.now();
    try {
      await this.reload();
      while (startTime + timeoutInSeconds * 1000 > Date.now()) {
        await this._delay(intervalInMilliseconds);
        await this.reload();
      }
      throw new _timeoutError2.default('Timed out waiting for account ' + this.id + ' to be deleted');
    } catch (err) {
      if (err.name === 'NotFoundError') {
        return;
      } else {
        throw err;
      }
    }
  }

  /**
   * Waits until API server has connected to the terminal and terminal has connected to the broker
   * @param {Number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {Number} intervalInMilliseconds interval between account reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when API server is connected to the broker
   * @throws {TimeoutError} if account have not connected to the broker withing timeout allowed
   */
  async waitConnected(timeoutInSeconds = 300, intervalInMilliseconds = 1000) {
    let startTime = Date.now();
    await this.reload();
    while (this.connectionStatus !== 'CONNECTED' && startTime + timeoutInSeconds * 1000 > Date.now()) {
      await this._delay(intervalInMilliseconds);
      await this.reload();
    }
    if (this.connectionStatus !== 'CONNECTED') {
      throw new _timeoutError2.default('Timed out waiting for account ' + this.id + ' to connect to the broker');
    }
  }

  /**
   * Connects to MetaApi. There is only one connection per account. Subsequent calls to this method will return the same connection.
   * @param {HistoryStorage} historyStorage optional history storage
   * @param {Date} [historyStartTime] history start time. Used for tests
   * @return {StreamingMetaApiConnection} MetaApi connection
   */
  getStreamingConnection(historyStorage, historyStartTime) {
    return this._connectionRegistry.connect(this, historyStorage, historyStartTime);
  }

  /**
   * Connects to MetaApi via RPC connection.
   * @returns {RpcMetaApiConnection} MetaApi connection
   */
  getRPCConnection() {
    return new _rpcMetaApiConnection2.default(this._metaApiWebsocketClient, this);
  }

  /**
   * Updates MetaTrader account data
   * @param {MetatraderAccountUpdateDto} account MetaTrader account update
   * @return {Promise} promise resolving when account is updated
   */
  async update(account) {
    await this._metatraderAccountClient.updateAccount(this.id, account);
    await this.reload();
  }

  /**
   * Retrieves expert advisor of current account
   * @returns {Promise<ExpertAdvisor[]>} promise resolving with an array of expert advisor entities
   */
  async getExpertAdvisors() {
    this._checkExpertAdvisorAllowed();
    let expertAdvisors = await this._expertAdvisorClient.getExpertAdvisors(this.id);
    return expertAdvisors.map(e => new _expertAdvisor2.default(e, this.id, this._expertAdvisorClient));
  }

  /**
   * Retrieves a expert advisor of current account by id
   * @param {String} expertId expert advisor id
   * @returns {Promise<ExpertAdvisor>} promise resolving with expert advisor entity
   */
  async getExpertAdvisor(expertId) {
    this._checkExpertAdvisorAllowed();
    let expertAdvisor = await this._expertAdvisorClient.getExpertAdvisor(this.id, expertId);
    return new _expertAdvisor2.default(expertAdvisor, this.id, this._expertAdvisorClient);
  }

  /**
   * Creates an expert advisor
   * @param {String} expertId expert advisor id
   * @param {NewExpertAdvisorDto} expert expert advisor data
   * @returns {Promise<ExpertAdvisor>} promise resolving with expert advisor entity
   */
  async createExpertAdvisor(expertId, expert) {
    this._checkExpertAdvisorAllowed();
    await this._expertAdvisorClient.updateExpertAdvisor(this.id, expertId, expert);
    return this.getExpertAdvisor(expertId);
  }

  /**
   * Returns historical candles for a specific symbol and timeframe from the MetaTrader account.
   * See https://metaapi.cloud/docs/client/restApi/api/retrieveMarketData/readHistoricalCandles/
   * @param {string} symbol symbol to retrieve candles for (e.g. a currency pair or an index)
   * @param {string} timeframe defines the timeframe according to which the candles must be generated. Allowed values
   * for MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed
   * values for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {Date} [startTime] time to start loading candles from. Note that candles are loaded in backwards direction, so
   * this should be the latest time. Leave empty to request latest candles.
   * @param {number} [limit] maximum number of candles to retrieve. Must be less or equal to 1000
   * @return {Promise<Array<MetatraderCandle>>} promise resolving with historical candles downloaded
   */
  getHistoricalCandles(symbol, timeframe, startTime, limit) {
    return this._historicalMarketDataClient.getHistoricalCandles(this.id, symbol, timeframe, startTime, limit);
  }

  /**
   * Returns historical ticks for a specific symbol from the MetaTrader account. This API is not supported by MT4
   * accounts.
   * See https://metaapi.cloud/docs/client/restApi/api/retrieveMarketData/readHistoricalTicks/
   * @param {string} symbol symbol to retrieve ticks for (e.g. a currency pair or an index)
   * @param {Date} [startTime] time to start loading ticks from. Note that candles are loaded in forward direction, so
   * this should be the earliest time. Leave empty to request latest candles.
   * @param {number} [offset] number of ticks to skip (you can use it to avoid requesting ticks from previous request
   * twice)
   * @param {number} [limit] maximum number of ticks to retrieve. Must be less or equal to 1000
   * @return {Promise<Array<MetatraderTick>>} promise resolving with historical ticks downloaded
   */
  getHistoricalTicks(symbol, startTime, offset, limit) {
    return this._historicalMarketDataClient.getHistoricalTicks(this.id, symbol, startTime, offset, limit);
  }

  _checkExpertAdvisorAllowed() {
    if (this.version !== 4 || this.type !== 'cloud-g1') {
      throw new _errorHandler.ValidationError('Custom expert advisor is available only for MT4 G1 accounts');
    }
  }

  _delay(timeoutInMilliseconds) {
    return new _promise2.default(res => setTimeout(res, timeoutInMilliseconds));
  }

}
exports.default = MetatraderAccount;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL2xpYi9tZXRhQXBpL21ldGF0cmFkZXJBY2NvdW50LmVzNiJdLCJuYW1lcyI6WyJNZXRhdHJhZGVyQWNjb3VudCIsImNvbnN0cnVjdG9yIiwiZGF0YSIsIm1ldGF0cmFkZXJBY2NvdW50Q2xpZW50IiwibWV0YUFwaVdlYnNvY2tldENsaWVudCIsImNvbm5lY3Rpb25SZWdpc3RyeSIsImV4cGVydEFkdmlzb3JDbGllbnQiLCJoaXN0b3JpY2FsTWFya2V0RGF0YUNsaWVudCIsIl9kYXRhIiwiX21ldGF0cmFkZXJBY2NvdW50Q2xpZW50IiwiX21ldGFBcGlXZWJzb2NrZXRDbGllbnQiLCJfY29ubmVjdGlvblJlZ2lzdHJ5IiwiX2V4cGVydEFkdmlzb3JDbGllbnQiLCJfaGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnQiLCJpZCIsIl9pZCIsIm5hbWUiLCJ0eXBlIiwibG9naW4iLCJzZXJ2ZXIiLCJwcm92aXNpb25pbmdQcm9maWxlSWQiLCJhcHBsaWNhdGlvbiIsIm1hZ2ljIiwic3RhdGUiLCJjb25uZWN0aW9uU3RhdHVzIiwiYWNjZXNzVG9rZW4iLCJtYW51YWxUcmFkZXMiLCJleHRlbnNpb25zIiwibWV0YWRhdGEiLCJ0YWdzIiwiY29weUZhY3RvcnlSb2xlcyIsInJlc291cmNlU2xvdHMiLCJjb3B5RmFjdG9yeVJlc291cmNlU2xvdHMiLCJiYXNlQ3VycmVuY3kiLCJyZWxpYWJpbGl0eSIsInZlcnNpb24iLCJyZWxvYWQiLCJnZXRBY2NvdW50IiwicmVtb3ZlIiwiZGVsZXRlQWNjb3VudCIsImZpbGVNYW5hZ2VyIiwiSGlzdG9yeUZpbGVNYW5hZ2VyIiwiZGVsZXRlU3RvcmFnZUZyb21EaXNrIiwiZXJyIiwiZGVwbG95IiwiZGVwbG95QWNjb3VudCIsInVuZGVwbG95IiwidW5kZXBsb3lBY2NvdW50IiwicmVkZXBsb3kiLCJyZWRlcGxveUFjY291bnQiLCJpbmNyZWFzZVJlbGlhYmlsaXR5Iiwid2FpdERlcGxveWVkIiwidGltZW91dEluU2Vjb25kcyIsImludGVydmFsSW5NaWxsaXNlY29uZHMiLCJzdGFydFRpbWUiLCJEYXRlIiwibm93IiwiX2RlbGF5IiwiVGltZW91dEVycm9yIiwid2FpdFVuZGVwbG95ZWQiLCJ3YWl0UmVtb3ZlZCIsIndhaXRDb25uZWN0ZWQiLCJnZXRTdHJlYW1pbmdDb25uZWN0aW9uIiwiaGlzdG9yeVN0b3JhZ2UiLCJoaXN0b3J5U3RhcnRUaW1lIiwiY29ubmVjdCIsImdldFJQQ0Nvbm5lY3Rpb24iLCJScGNNZXRhQXBpQ29ubmVjdGlvbiIsInVwZGF0ZSIsImFjY291bnQiLCJ1cGRhdGVBY2NvdW50IiwiZ2V0RXhwZXJ0QWR2aXNvcnMiLCJfY2hlY2tFeHBlcnRBZHZpc29yQWxsb3dlZCIsImV4cGVydEFkdmlzb3JzIiwibWFwIiwiZSIsIkV4cGVydEFkdmlzb3IiLCJnZXRFeHBlcnRBZHZpc29yIiwiZXhwZXJ0SWQiLCJleHBlcnRBZHZpc29yIiwiY3JlYXRlRXhwZXJ0QWR2aXNvciIsImV4cGVydCIsInVwZGF0ZUV4cGVydEFkdmlzb3IiLCJnZXRIaXN0b3JpY2FsQ2FuZGxlcyIsInN5bWJvbCIsInRpbWVmcmFtZSIsImxpbWl0IiwiZ2V0SGlzdG9yaWNhbFRpY2tzIiwib2Zmc2V0IiwiVmFsaWRhdGlvbkVycm9yIiwidGltZW91dEluTWlsbGlzZWNvbmRzIiwicmVzIiwic2V0VGltZW91dCJdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7Ozs7Ozs7QUFFQTs7OztBQUNBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBRUE7OztBQUdlLE1BQU1BLGlCQUFOLENBQXdCOztBQUVyQzs7Ozs7Ozs7O0FBU0FDLGNBQVlDLElBQVosRUFBa0JDLHVCQUFsQixFQUEyQ0Msc0JBQTNDLEVBQW1FQyxrQkFBbkUsRUFBdUZDLG1CQUF2RixFQUNFQywwQkFERixFQUM4QjtBQUM1QixTQUFLQyxLQUFMLEdBQWFOLElBQWI7QUFDQSxTQUFLTyx3QkFBTCxHQUFnQ04sdUJBQWhDO0FBQ0EsU0FBS08sdUJBQUwsR0FBK0JOLHNCQUEvQjtBQUNBLFNBQUtPLG1CQUFMLEdBQTJCTixrQkFBM0I7QUFDQSxTQUFLTyxvQkFBTCxHQUE0Qk4sbUJBQTVCO0FBQ0EsU0FBS08sMkJBQUwsR0FBbUNOLDBCQUFuQztBQUNEOztBQUVEOzs7O0FBSUEsTUFBSU8sRUFBSixHQUFTO0FBQ1AsV0FBTyxLQUFLTixLQUFMLENBQVdPLEdBQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJQyxJQUFKLEdBQVc7QUFDVCxXQUFPLEtBQUtSLEtBQUwsQ0FBV1EsSUFBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLElBQUosR0FBVztBQUNULFdBQU8sS0FBS1QsS0FBTCxDQUFXUyxJQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsS0FBSixHQUFZO0FBQ1YsV0FBTyxLQUFLVixLQUFMLENBQVdVLEtBQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJQyxNQUFKLEdBQWE7QUFDWCxXQUFPLEtBQUtYLEtBQUwsQ0FBV1csTUFBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLHFCQUFKLEdBQTRCO0FBQzFCLFdBQU8sS0FBS1osS0FBTCxDQUFXWSxxQkFBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLFdBQUosR0FBa0I7QUFDaEIsV0FBTyxLQUFLYixLQUFMLENBQVdhLFdBQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJQyxLQUFKLEdBQVk7QUFDVixXQUFPLEtBQUtkLEtBQUwsQ0FBV2MsS0FBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLEtBQUosR0FBWTtBQUNWLFdBQU8sS0FBS2YsS0FBTCxDQUFXZSxLQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsZ0JBQUosR0FBdUI7QUFDckIsV0FBTyxLQUFLaEIsS0FBTCxDQUFXZ0IsZ0JBQWxCO0FBQ0Q7O0FBRUQ7Ozs7O0FBS0EsTUFBSUMsV0FBSixHQUFrQjtBQUNoQixXQUFPLEtBQUtqQixLQUFMLENBQVdpQixXQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsWUFBSixHQUFtQjtBQUNqQixXQUFPLENBQUMsQ0FBQyxLQUFLbEIsS0FBTCxDQUFXa0IsWUFBcEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLFVBQUosR0FBaUI7QUFDZixXQUFPLEtBQUtuQixLQUFMLENBQVdtQixVQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsUUFBSixHQUFlO0FBQ2IsV0FBTyxLQUFLcEIsS0FBTCxDQUFXb0IsUUFBbEI7QUFDRDs7QUFFRDs7OztBQUlBLE1BQUlDLElBQUosR0FBVztBQUNULFdBQU8sS0FBS3JCLEtBQUwsQ0FBV3FCLElBQWxCO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxNQUFJQyxnQkFBSixHQUF1QjtBQUNyQixXQUFPLEtBQUt0QixLQUFMLENBQVdzQixnQkFBbEI7QUFDRDs7QUFFRDs7Ozs7Ozs7O0FBU0EsTUFBSUMsYUFBSixHQUFvQjtBQUNsQixXQUFPLEtBQUt2QixLQUFMLENBQVd1QixhQUFsQjtBQUNEOztBQUVEOzs7Ozs7Ozs7QUFTQSxNQUFJQyx3QkFBSixHQUErQjtBQUM3QixXQUFPLEtBQUt4QixLQUFMLENBQVd3Qix3QkFBbEI7QUFDRDs7QUFFRDs7Ozs7O0FBTUEsTUFBSUMsWUFBSixHQUFtQjtBQUNqQixXQUFPLEtBQUt6QixLQUFMLENBQVd5QixZQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsV0FBSixHQUFrQjtBQUNoQixXQUFPLEtBQUsxQixLQUFMLENBQVcwQixXQUFsQjtBQUNEOztBQUVEOzs7O0FBSUEsTUFBSUMsT0FBSixHQUFjO0FBQ1osV0FBTyxLQUFLM0IsS0FBTCxDQUFXMkIsT0FBbEI7QUFDRDs7QUFFRDs7OztBQUlBLFFBQU1DLE1BQU4sR0FBZTtBQUNiLFNBQUs1QixLQUFMLEdBQWEsTUFBTSxLQUFLQyx3QkFBTCxDQUE4QjRCLFVBQTlCLENBQXlDLEtBQUt2QixFQUE5QyxDQUFuQjtBQUNEOztBQUVEOzs7Ozs7QUFNQSxRQUFNd0IsTUFBTixHQUFlO0FBQ2IsU0FBSzNCLG1CQUFMLENBQXlCMkIsTUFBekIsQ0FBZ0MsS0FBS3hCLEVBQXJDO0FBQ0EsVUFBTSxLQUFLTCx3QkFBTCxDQUE4QjhCLGFBQTlCLENBQTRDLEtBQUt6QixFQUFqRCxDQUFOO0FBQ0EsVUFBTTBCLGNBQWMsSUFBSUMsZUFBSixDQUF1QixLQUFLM0IsRUFBNUIsQ0FBcEI7QUFDQSxVQUFNMEIsWUFBWUUscUJBQVosRUFBTjtBQUNBLFFBQUksS0FBS3pCLElBQUwsS0FBYyxhQUFsQixFQUFpQztBQUMvQixVQUFJO0FBQ0YsY0FBTSxLQUFLbUIsTUFBTCxFQUFOO0FBQ0QsT0FGRCxDQUVFLE9BQU9PLEdBQVAsRUFBWTtBQUNaLFlBQUlBLElBQUkzQixJQUFKLEtBQWEsZUFBakIsRUFBa0M7QUFDaEMsZ0JBQU0yQixHQUFOO0FBQ0Q7QUFDRjtBQUNGO0FBQ0Y7O0FBRUQ7Ozs7O0FBS0EsUUFBTUMsTUFBTixHQUFlO0FBQ2IsVUFBTSxLQUFLbkMsd0JBQUwsQ0FBOEJvQyxhQUE5QixDQUE0QyxLQUFLL0IsRUFBakQsQ0FBTjtBQUNBLFVBQU0sS0FBS3NCLE1BQUwsRUFBTjtBQUNEOztBQUVEOzs7OztBQUtBLFFBQU1VLFFBQU4sR0FBaUI7QUFDZixTQUFLbkMsbUJBQUwsQ0FBeUIyQixNQUF6QixDQUFnQyxLQUFLeEIsRUFBckM7QUFDQSxVQUFNLEtBQUtMLHdCQUFMLENBQThCc0MsZUFBOUIsQ0FBOEMsS0FBS2pDLEVBQW5ELENBQU47QUFDQSxVQUFNLEtBQUtzQixNQUFMLEVBQU47QUFDRDs7QUFFRDs7Ozs7QUFLQSxRQUFNWSxRQUFOLEdBQWlCO0FBQ2YsVUFBTSxLQUFLdkMsd0JBQUwsQ0FBOEJ3QyxlQUE5QixDQUE4QyxLQUFLbkMsRUFBbkQsQ0FBTjtBQUNBLFVBQU0sS0FBS3NCLE1BQUwsRUFBTjtBQUNEOztBQUVEOzs7O0FBSUEsUUFBTWMsbUJBQU4sR0FBNEI7QUFDMUIsVUFBTSxLQUFLekMsd0JBQUwsQ0FBOEJ5QyxtQkFBOUIsQ0FBa0QsS0FBS3BDLEVBQXZELENBQU47QUFDQSxVQUFNLEtBQUtzQixNQUFMLEVBQU47QUFDRDs7QUFFRDs7Ozs7OztBQU9BLFFBQU1lLFlBQU4sQ0FBbUJDLG1CQUFtQixHQUF0QyxFQUEyQ0MseUJBQXlCLElBQXBFLEVBQTBFO0FBQ3hFLFFBQUlDLFlBQVlDLEtBQUtDLEdBQUwsRUFBaEI7QUFDQSxVQUFNLEtBQUtwQixNQUFMLEVBQU47QUFDQSxXQUFPLEtBQUtiLEtBQUwsS0FBZSxVQUFmLElBQThCK0IsWUFBWUYsbUJBQW1CLElBQWhDLEdBQXdDRyxLQUFLQyxHQUFMLEVBQTVFLEVBQXdGO0FBQ3RGLFlBQU0sS0FBS0MsTUFBTCxDQUFZSixzQkFBWixDQUFOO0FBQ0EsWUFBTSxLQUFLakIsTUFBTCxFQUFOO0FBQ0Q7QUFDRCxRQUFJLEtBQUtiLEtBQUwsS0FBZSxVQUFuQixFQUErQjtBQUM3QixZQUFNLElBQUltQyxzQkFBSixDQUFpQixtQ0FBbUMsS0FBSzVDLEVBQXhDLEdBQTZDLGlCQUE5RCxDQUFOO0FBQ0Q7QUFDRjs7QUFFRDs7Ozs7OztBQU9BLFFBQU02QyxjQUFOLENBQXFCUCxtQkFBbUIsR0FBeEMsRUFBNkNDLHlCQUF5QixJQUF0RSxFQUE0RTtBQUMxRSxRQUFJQyxZQUFZQyxLQUFLQyxHQUFMLEVBQWhCO0FBQ0EsVUFBTSxLQUFLcEIsTUFBTCxFQUFOO0FBQ0EsV0FBTyxLQUFLYixLQUFMLEtBQWUsWUFBZixJQUFnQytCLFlBQVlGLG1CQUFtQixJQUFoQyxHQUF3Q0csS0FBS0MsR0FBTCxFQUE5RSxFQUEwRjtBQUN4RixZQUFNLEtBQUtDLE1BQUwsQ0FBWUosc0JBQVosQ0FBTjtBQUNBLFlBQU0sS0FBS2pCLE1BQUwsRUFBTjtBQUNEO0FBQ0QsUUFBSSxLQUFLYixLQUFMLEtBQWUsWUFBbkIsRUFBaUM7QUFDL0IsWUFBTSxJQUFJbUMsc0JBQUosQ0FBaUIsbUNBQW1DLEtBQUs1QyxFQUF4QyxHQUE2QyxtQkFBOUQsQ0FBTjtBQUNEO0FBQ0Y7O0FBRUQ7Ozs7Ozs7QUFPQSxRQUFNOEMsV0FBTixDQUFrQlIsbUJBQW1CLEdBQXJDLEVBQTBDQyx5QkFBeUIsSUFBbkUsRUFBeUU7QUFDdkUsUUFBSUMsWUFBWUMsS0FBS0MsR0FBTCxFQUFoQjtBQUNBLFFBQUk7QUFDRixZQUFNLEtBQUtwQixNQUFMLEVBQU47QUFDQSxhQUFPa0IsWUFBWUYsbUJBQW1CLElBQS9CLEdBQXNDRyxLQUFLQyxHQUFMLEVBQTdDLEVBQXlEO0FBQ3ZELGNBQU0sS0FBS0MsTUFBTCxDQUFZSixzQkFBWixDQUFOO0FBQ0EsY0FBTSxLQUFLakIsTUFBTCxFQUFOO0FBQ0Q7QUFDRCxZQUFNLElBQUlzQixzQkFBSixDQUFpQixtQ0FBbUMsS0FBSzVDLEVBQXhDLEdBQTZDLGdCQUE5RCxDQUFOO0FBQ0QsS0FQRCxDQU9FLE9BQU82QixHQUFQLEVBQVk7QUFDWixVQUFJQSxJQUFJM0IsSUFBSixLQUFhLGVBQWpCLEVBQWtDO0FBQ2hDO0FBQ0QsT0FGRCxNQUVPO0FBQ0wsY0FBTTJCLEdBQU47QUFDRDtBQUNGO0FBQ0Y7O0FBRUQ7Ozs7Ozs7QUFPQSxRQUFNa0IsYUFBTixDQUFvQlQsbUJBQW1CLEdBQXZDLEVBQTRDQyx5QkFBeUIsSUFBckUsRUFBMkU7QUFDekUsUUFBSUMsWUFBWUMsS0FBS0MsR0FBTCxFQUFoQjtBQUNBLFVBQU0sS0FBS3BCLE1BQUwsRUFBTjtBQUNBLFdBQU8sS0FBS1osZ0JBQUwsS0FBMEIsV0FBMUIsSUFBMEM4QixZQUFZRixtQkFBbUIsSUFBaEMsR0FBd0NHLEtBQUtDLEdBQUwsRUFBeEYsRUFBb0c7QUFDbEcsWUFBTSxLQUFLQyxNQUFMLENBQVlKLHNCQUFaLENBQU47QUFDQSxZQUFNLEtBQUtqQixNQUFMLEVBQU47QUFDRDtBQUNELFFBQUksS0FBS1osZ0JBQUwsS0FBMEIsV0FBOUIsRUFBMkM7QUFDekMsWUFBTSxJQUFJa0Msc0JBQUosQ0FBaUIsbUNBQW1DLEtBQUs1QyxFQUF4QyxHQUE2QywyQkFBOUQsQ0FBTjtBQUNEO0FBQ0Y7O0FBRUQ7Ozs7OztBQU1BZ0QseUJBQXVCQyxjQUF2QixFQUF1Q0MsZ0JBQXZDLEVBQXlEO0FBQ3ZELFdBQU8sS0FBS3JELG1CQUFMLENBQXlCc0QsT0FBekIsQ0FBaUMsSUFBakMsRUFBdUNGLGNBQXZDLEVBQXVEQyxnQkFBdkQsQ0FBUDtBQUNEOztBQUVEOzs7O0FBSUFFLHFCQUFtQjtBQUNqQixXQUFPLElBQUlDLDhCQUFKLENBQXlCLEtBQUt6RCx1QkFBOUIsRUFBdUQsSUFBdkQsQ0FBUDtBQUNEOztBQUVEOzs7OztBQUtBLFFBQU0wRCxNQUFOLENBQWFDLE9BQWIsRUFBc0I7QUFDcEIsVUFBTSxLQUFLNUQsd0JBQUwsQ0FBOEI2RCxhQUE5QixDQUE0QyxLQUFLeEQsRUFBakQsRUFBcUR1RCxPQUFyRCxDQUFOO0FBQ0EsVUFBTSxLQUFLakMsTUFBTCxFQUFOO0FBQ0Q7O0FBRUQ7Ozs7QUFJQSxRQUFNbUMsaUJBQU4sR0FBMEI7QUFDeEIsU0FBS0MsMEJBQUw7QUFDQSxRQUFJQyxpQkFBaUIsTUFBTSxLQUFLN0Qsb0JBQUwsQ0FBMEIyRCxpQkFBMUIsQ0FBNEMsS0FBS3pELEVBQWpELENBQTNCO0FBQ0EsV0FBTzJELGVBQWVDLEdBQWYsQ0FBbUJDLEtBQUssSUFBSUMsdUJBQUosQ0FBa0JELENBQWxCLEVBQXFCLEtBQUs3RCxFQUExQixFQUE4QixLQUFLRixvQkFBbkMsQ0FBeEIsQ0FBUDtBQUNEOztBQUVEOzs7OztBQUtBLFFBQU1pRSxnQkFBTixDQUF1QkMsUUFBdkIsRUFBaUM7QUFDL0IsU0FBS04sMEJBQUw7QUFDQSxRQUFJTyxnQkFBZ0IsTUFBTSxLQUFLbkUsb0JBQUwsQ0FBMEJpRSxnQkFBMUIsQ0FBMkMsS0FBSy9ELEVBQWhELEVBQW9EZ0UsUUFBcEQsQ0FBMUI7QUFDQSxXQUFPLElBQUlGLHVCQUFKLENBQWtCRyxhQUFsQixFQUFpQyxLQUFLakUsRUFBdEMsRUFBMEMsS0FBS0Ysb0JBQS9DLENBQVA7QUFDRDs7QUFFRDs7Ozs7O0FBTUEsUUFBTW9FLG1CQUFOLENBQTBCRixRQUExQixFQUFvQ0csTUFBcEMsRUFBNEM7QUFDMUMsU0FBS1QsMEJBQUw7QUFDQSxVQUFNLEtBQUs1RCxvQkFBTCxDQUEwQnNFLG1CQUExQixDQUE4QyxLQUFLcEUsRUFBbkQsRUFBdURnRSxRQUF2RCxFQUFpRUcsTUFBakUsQ0FBTjtBQUNBLFdBQU8sS0FBS0osZ0JBQUwsQ0FBc0JDLFFBQXRCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7Ozs7O0FBWUFLLHVCQUFxQkMsTUFBckIsRUFBNkJDLFNBQTdCLEVBQXdDL0IsU0FBeEMsRUFBbURnQyxLQUFuRCxFQUEwRDtBQUN4RCxXQUFPLEtBQUt6RSwyQkFBTCxDQUFpQ3NFLG9CQUFqQyxDQUFzRCxLQUFLckUsRUFBM0QsRUFBK0RzRSxNQUEvRCxFQUF1RUMsU0FBdkUsRUFBa0YvQixTQUFsRixFQUE2RmdDLEtBQTdGLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7Ozs7O0FBWUFDLHFCQUFtQkgsTUFBbkIsRUFBMkI5QixTQUEzQixFQUFzQ2tDLE1BQXRDLEVBQThDRixLQUE5QyxFQUFxRDtBQUNuRCxXQUFPLEtBQUt6RSwyQkFBTCxDQUFpQzBFLGtCQUFqQyxDQUFvRCxLQUFLekUsRUFBekQsRUFBNkRzRSxNQUE3RCxFQUFxRTlCLFNBQXJFLEVBQWdGa0MsTUFBaEYsRUFBd0ZGLEtBQXhGLENBQVA7QUFDRDs7QUFFRGQsK0JBQTZCO0FBQzNCLFFBQUksS0FBS3JDLE9BQUwsS0FBaUIsQ0FBakIsSUFBc0IsS0FBS2xCLElBQUwsS0FBYyxVQUF4QyxFQUFvRDtBQUNsRCxZQUFNLElBQUl3RSw2QkFBSixDQUFvQiw2REFBcEIsQ0FBTjtBQUNEO0FBQ0Y7O0FBRURoQyxTQUFPaUMscUJBQVAsRUFBOEI7QUFDNUIsV0FBTyxzQkFBWUMsT0FBT0MsV0FBV0QsR0FBWCxFQUFnQkQscUJBQWhCLENBQW5CLENBQVA7QUFDRDs7QUF2Y29DO2tCQUFsQjFGLGlCIiwiZmlsZSI6Im1ldGF0cmFkZXJBY2NvdW50LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5pbXBvcnQgVGltZW91dEVycm9yIGZyb20gJy4uL2NsaWVudHMvdGltZW91dEVycm9yJztcbmltcG9ydCBScGNNZXRhQXBpQ29ubmVjdGlvbiBmcm9tICcuL3JwY01ldGFBcGlDb25uZWN0aW9uJztcbmltcG9ydCBIaXN0b3J5RmlsZU1hbmFnZXIgZnJvbSAnLi9oaXN0b3J5RmlsZU1hbmFnZXIvaW5kZXgnO1xuaW1wb3J0IEV4cGVydEFkdmlzb3IgZnJvbSAnLi9leHBlcnRBZHZpc29yJztcbmltcG9ydCB7VmFsaWRhdGlvbkVycm9yfSBmcm9tICcuLi9jbGllbnRzL2Vycm9ySGFuZGxlcic7XG5cbi8qKlxuICogSW1wbGVtZW50cyBhIE1ldGFUcmFkZXIgYWNjb3VudCBlbnRpdHlcbiAqL1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgTWV0YXRyYWRlckFjY291bnQge1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgTWV0YVRyYWRlciBhY2NvdW50IGVudGl0eVxuICAgKiBAcGFyYW0ge01ldGF0cmFkZXJBY2NvdW50RHRvfSBkYXRhIE1ldGFUcmFkZXIgYWNjb3VudCBkYXRhXG4gICAqIEBwYXJhbSB7TWV0YXRyYWRlckFjY291bnRDbGllbnR9IG1ldGF0cmFkZXJBY2NvdW50Q2xpZW50IE1ldGFUcmFkZXIgYWNjb3VudCBSRVNUIEFQSSBjbGllbnRcbiAgICogQHBhcmFtIHtNZXRhQXBpV2Vic29ja2V0Q2xpZW50fSBtZXRhQXBpV2Vic29ja2V0Q2xpZW50IE1ldGFBcGkgd2Vic29ja2V0IGNsaWVudFxuICAgKiBAcGFyYW0ge0Nvbm5lY3Rpb25SZWdpc3RyeX0gY29ubmVjdGlvblJlZ2lzdHJ5IG1ldGF0cmFkZXIgYWNjb3VudCBjb25uZWN0aW9uIHJlZ2lzdHJ5XG4gICAqIEBwYXJhbSB7RXhwZXJ0QWR2aXNvckNsaWVudH0gZXhwZXJ0QWR2aXNvckNsaWVudCBleHBlcnQgYWR2aXNvciBSRVNUIEFQSSBjbGllbnRcbiAgICogQHBhcmFtIHtIaXN0b3JpY2FsTWFya2V0RGF0YUNsaWVudH0gaGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnQgaGlzdG9yaWNhbCBtYXJrZXQgZGF0YSBIVFRQIEFQSSBjbGllbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yKGRhdGEsIG1ldGF0cmFkZXJBY2NvdW50Q2xpZW50LCBtZXRhQXBpV2Vic29ja2V0Q2xpZW50LCBjb25uZWN0aW9uUmVnaXN0cnksIGV4cGVydEFkdmlzb3JDbGllbnQsIFxuICAgIGhpc3RvcmljYWxNYXJrZXREYXRhQ2xpZW50KSB7XG4gICAgdGhpcy5fZGF0YSA9IGRhdGE7XG4gICAgdGhpcy5fbWV0YXRyYWRlckFjY291bnRDbGllbnQgPSBtZXRhdHJhZGVyQWNjb3VudENsaWVudDtcbiAgICB0aGlzLl9tZXRhQXBpV2Vic29ja2V0Q2xpZW50ID0gbWV0YUFwaVdlYnNvY2tldENsaWVudDtcbiAgICB0aGlzLl9jb25uZWN0aW9uUmVnaXN0cnkgPSBjb25uZWN0aW9uUmVnaXN0cnk7XG4gICAgdGhpcy5fZXhwZXJ0QWR2aXNvckNsaWVudCA9IGV4cGVydEFkdmlzb3JDbGllbnQ7XG4gICAgdGhpcy5faGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnQgPSBoaXN0b3JpY2FsTWFya2V0RGF0YUNsaWVudDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFjY291bnQgaWRcbiAgICogQHJldHVybiB7U3RyaW5nfSBhY2NvdW50IGlkXG4gICAqL1xuICBnZXQgaWQoKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RhdGEuX2lkO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYWNjb3VudCBuYW1lXG4gICAqIEByZXR1cm4ge1N0cmluZ30gYWNjb3VudCBuYW1lXG4gICAqL1xuICBnZXQgbmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5uYW1lO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYWNjb3VudCB0eXBlLiBQb3NzaWJsZSB2YWx1ZXMgYXJlIGNsb3VkLCBjbG91ZC1nMSwgY2xvdWQtZzIgYW5kIHNlbGYtaG9zdGVkLlxuICAgKiBAcmV0dXJuIHtTdHJpbmd9IGFjY291bnQgdHlwZVxuICAgKi9cbiAgZ2V0IHR5cGUoKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RhdGEudHlwZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFjY291bnQgbG9naW5cbiAgICogQHJldHVybiB7U3RyaW5nfSBhY2NvdW50IGxvZ2luXG4gICAqL1xuICBnZXQgbG9naW4oKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RhdGEubG9naW47XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBNZXRhVHJhZGVyIHNlcnZlciB3aGljaCBob3N0cyB0aGUgYWNjb3VudFxuICAgKiBAcmV0dXJuIHtTdHJpbmd9IE1ldGFUcmFkZXIgc2VydmVyIHdoaWNoIGhvc3RzIHRoZSBhY2NvdW50XG4gICAqL1xuICBnZXQgc2VydmVyKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLnNlcnZlcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGlkIG9mIHRoZSBhY2NvdW50J3MgcHJvdmlzaW9uaW5nIHByb2ZpbGVcbiAgICogQHJldHVybiB7U3RyaW5nfSBpZCBvZiB0aGUgYWNjb3VudCdzIHByb3Zpc2lvbmluZyBwcm9maWxlXG4gICAqL1xuICBnZXQgcHJvdmlzaW9uaW5nUHJvZmlsZUlkKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLnByb3Zpc2lvbmluZ1Byb2ZpbGVJZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFwcGxpY2F0aW9uIG5hbWUgdG8gY29ubmVjdCB0aGUgYWNjb3VudCB0by4gQ3VycmVudGx5IGFsbG93ZWQgdmFsdWVzIGFyZSBNZXRhQXBpIGFuZCBBZ2lsaXVtVHJhZGVcbiAgICogQHJldHVybiB7U3RyaW5nfSBhcHBsaWNhdGlvbiBuYW1lIHRvIGNvbm5lY3QgdGhlIGFjY291bnQgdG9cbiAgICovXG4gIGdldCBhcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5hcHBsaWNhdGlvbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIE1ldGFUcmFkZXIgbWFnaWMgdG8gcGxhY2UgdHJhZGVzIHVzaW5nXG4gICAqIEByZXR1cm4ge051bWJlcn0gTWV0YVRyYWRlciBtYWdpYyB0byBwbGFjZSB0cmFkZXMgdXNpbmdcbiAgICovXG4gIGdldCBtYWdpYygpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5tYWdpYztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFjY291bnQgZGVwbG95bWVudCBzdGF0ZS4gT25lIG9mIENSRUFURUQsIERFUExPWUlORywgREVQTE9ZRUQsIFVOREVQTE9ZSU5HLCBVTkRFUExPWUVELCBERUxFVElOR1xuICAgKiBAcmV0dXJuIHtTdHJpbmd9IGFjY291bnQgZGVwbG95bWVudCBzdGF0ZVxuICAgKi9cbiAgZ2V0IHN0YXRlKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLnN0YXRlO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGVybWluYWwgJiBicm9rZXIgY29ubmVjdGlvbiBzdGF0dXMsIG9uZSBvZiBDT05ORUNURUQsIERJU0NPTk5FQ1RFRCwgRElTQ09OTkVDVEVEX0ZST01fQlJPS0VSXG4gICAqIEByZXR1cm4ge1N0cmluZ30gdGVybWluYWwgJiBicm9rZXIgY29ubmVjdGlvbiBzdGF0dXNcbiAgICovXG4gIGdldCBjb25uZWN0aW9uU3RhdHVzKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLmNvbm5lY3Rpb25TdGF0dXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhdXRob3JpemF0aW9uIGFjY2VzcyB0b2tlbiB0byBiZSB1c2VkIGZvciBhY2Nlc3Npbmcgc2luZ2xlIGFjY291bnQgZGF0YS5cbiAgICogSW50ZW5kZWQgdG8gYmUgdXNlZCBpbiBicm93c2VyIEFQSS5cbiAgICogQHJldHVybiB7U3RyaW5nfSBhdXRob3JpemF0aW9uIHRva2VuXG4gICAqL1xuICBnZXQgYWNjZXNzVG9rZW4oKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RhdGEuYWNjZXNzVG9rZW47XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBmbGFnIGluZGljYXRpbmcgaWYgdHJhZGVzIHNob3VsZCBiZSBwbGFjZWQgYXMgbWFudWFsIHRyYWRlcyBvbiB0aGlzIGFjY291bnRcbiAgICogQHJldHVybiB7Qm9vbGVhbn0gZmxhZyBpbmRpY2F0aW5nIGlmIHRyYWRlcyBzaG91bGQgYmUgcGxhY2VkIGFzIG1hbnVhbCB0cmFkZXMgb24gdGhpcyBhY2NvdW50XG4gICAqL1xuICBnZXQgbWFudWFsVHJhZGVzKCkge1xuICAgIHJldHVybiAhIXRoaXMuX2RhdGEubWFudWFsVHJhZGVzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgQVBJIGV4dGVuc2lvbnNcbiAgICogQHJldHVybiB7QXJyYXk8RXh0ZW5zaW9uPn0gQVBJIGV4dGVuc2lvbnNcbiAgICovXG4gIGdldCBleHRlbnNpb25zKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLmV4dGVuc2lvbnM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBleHRyYSBpbmZvcm1hdGlvbiB3aGljaCBjYW4gYmUgc3RvcmVkIHRvZ2V0aGVyIHdpdGggeW91ciBhY2NvdW50XG4gICAqIEByZXR1cm4ge09iamVjdH0gZXh0cmEgaW5mb3JtYXRpb24gd2hpY2ggY2FuIGJlIHN0b3JlZCB0b2dldGhlciB3aXRoIHlvdXIgYWNjb3VudFxuICAgKi9cbiAgZ2V0IG1ldGFkYXRhKCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLm1ldGFkYXRhO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdXNlci1kZWZpbmVkIGFjY291bnQgdGFnc1xuICAgKiBAcmV0dXJuIHtBcnJheTxzdHJpbmc+fSB1c2VyLWRlZmluZWQgYWNjb3VudCB0YWdzXG4gICAqL1xuICBnZXQgdGFncygpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS50YWdzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYWNjb3VudCByb2xlcyBmb3IgQ29weUZhY3RvcnkyIGFwcGxpY2F0aW9uXG4gICAqIEByZXR1cm4ge0FycmF5PHN0cmluZz59IGFjY291bnQgcm9sZXMgZm9yIENvcHlGYWN0b3J5MiBhcHBsaWNhdGlvblxuICAgKi9cbiAgZ2V0IGNvcHlGYWN0b3J5Um9sZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RhdGEuY29weUZhY3RvcnlSb2xlcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIG51bWJlciBvZiByZXNvdXJjZSBzbG90cyB0byBhbGxvY2F0ZSB0byBhY2NvdW50LiBBbGxvY2F0aW5nIGV4dHJhIHJlc291cmNlIHNsb3RzXG4gICAqIHJlc3VsdHMgaW4gYmV0dGVyIGFjY291bnQgcGVyZm9ybWFuY2UgdW5kZXIgbG9hZCB3aGljaCBpcyB1c2VmdWwgZm9yIHNvbWUgYXBwbGljYXRpb25zLiBFLmcuIGlmIHlvdSBoYXZlIG1hbnlcbiAgICogYWNjb3VudHMgY29weWluZyB0aGUgc2FtZSBzdHJhdGVneSB2aWEgQ29veUZhY3RvcnkgQVBJLCB0aGVuIHlvdSBjYW4gaW5jcmVhc2UgcmVzb3VyY2VTbG90cyB0byBnZXQgYSBsb3dlciB0cmFkZVxuICAgKiBjb3B5aW5nIGxhdGVuY3kuIFBsZWFzZSBub3RlIHRoYXQgYWxsb2NhdGluZyBleHRyYSByZXNvdXJjZSBzbG90cyBpcyBhIHBhaWQgb3B0aW9uLiBQbGVhc2Ugbm90ZSB0aGF0IGhpZ2hcbiAgICogcmVsaWFiaWxpdHkgYWNjb3VudHMgdXNlIHJlZHVuZGFudCBpbmZyYXN0cnVjdHVyZSwgc28gdGhhdCBlYWNoIHJlc291cmNlIHNsb3QgZm9yIGEgaGlnaCByZWxpYWJpbGl0eSBhY2NvdW50XG4gICAqIGlzIGJpbGxlZCBhcyAyIHN0YW5kYXJkIHJlc291cmNlIHNsb3RzLiAgRGVmYXVsdCBpcyAxLlxuICAgKiBAcmV0dXJuIHtudW1iZXJ9IG51bWJlciBvZiByZXNvdXJjZSBzbG90cyB0byBhbGxvY2F0ZSB0byBhY2NvdW50XG4gICAqL1xuICBnZXQgcmVzb3VyY2VTbG90cygpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5yZXNvdXJjZVNsb3RzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIG51bWJlciBvZiBDb3B5RmFjdG9yeSAyIHJlc291cmNlIHNsb3RzIHRvIGFsbG9jYXRlIHRvIGFjY291bnQuXG4gICAqIEFsbG9jYXRpbmcgZXh0cmEgcmVzb3VyY2Ugc2xvdHMgcmVzdWx0cyBpbiBsb3dlciB0cmFkZSBjb3B5aW5nIGxhdGVuY3kuIFBsZWFzZSBub3RlIHRoYXQgYWxsb2NhdGluZyBleHRyYSByZXNvdXJjZVxuICAgKiBzbG90cyBpcyBhIHBhaWQgb3B0aW9uLiBQbGVhc2UgYWxzbyBub3RlIHRoYXQgQ29weUZhY3RvcnkgMiB1c2VzIHJlZHVuZGFudCBpbmZyYXN0cnVjdHVyZSBzbyB0aGF0XG4gICAqIGVhY2ggQ29weUZhY3RvcnkgcmVzb3VyY2Ugc2xvdCBpcyBiaWxsZWQgYXMgMiBzdGFuZGFyZCByZXNvdXJjZSBzbG90cy4gWW91IHdpbGwgYmUgYmlsbGVkIGZvciBDb3B5RmFjdG9yeSAyXG4gICAqIHJlc291cmNlIHNsb3RzIG9ubHkgaWYgeW91IGhhdmUgYWRkZWQgeW91ciBhY2NvdW50IHRvIENvcHlGYWN0b3J5IDIgYnkgc3BlY2lmeWluZyBjb3B5RmFjdG9yeVJvbGVzIGZpZWxkLlxuICAgKiBEZWZhdWx0IGlzIDEuXG4gICAqIEByZXR1cm4ge251bWJlcn0gbnVtYmVyIG9mIENvcHlGYWN0b3J5IDIgcmVzb3VyY2Ugc2xvdHMgdG8gYWxsb2NhdGUgdG8gYWNjb3VudFxuICAgKi9cbiAgZ2V0IGNvcHlGYWN0b3J5UmVzb3VyY2VTbG90cygpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5jb3B5RmFjdG9yeVJlc291cmNlU2xvdHM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyAzLWNoYXJhY3RlciBJU08gY3VycmVuY3kgY29kZSBvZiB0aGUgYWNjb3VudCBiYXNlIGN1cnJlbmN5LiBEZWZhdWx0IHZhbHVlIGlzIFVTRC4gVGhlIHNldHRpbmcgaXMgdG8gYmUgdXNlZFxuICAgKiBmb3IgY29weSB0cmFkaW5nIGFjY291bnRzIHdoaWNoIHVzZSBuYXRpb25hbCBjdXJyZW5jaWVzIG9ubHksIHN1Y2ggYXMgc29tZSBCcmF6aWxpYW4gYnJva2Vycy4gWW91IHNob3VsZCBub3QgYWx0ZXJcbiAgICogdGhpcyBzZXR0aW5nIHVubGVzcyB5b3UgdW5kZXJzdGFuZCB3aGF0IHlvdSBhcmUgZG9pbmcuXG4gICAqIEByZXR1cm4ge251bWJlcn0gMy1jaGFyYWN0ZXIgSVNPIGN1cnJlbmN5IGNvZGUgb2YgdGhlIGFjY291bnQgYmFzZSBjdXJyZW5jeVxuICAgKi9cbiAgZ2V0IGJhc2VDdXJyZW5jeSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS5iYXNlQ3VycmVuY3k7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyByZWxpYWJpbGl0eSB2YWx1ZS4gUG9zc2libGUgdmFsdWVzIGFyZSByZWd1bGFyIGFuZCBoaWdoXG4gICAqIEByZXR1cm4ge1N0cmluZ30gYWNjb3VudCByZWxpYWJpbGl0eSB2YWx1ZVxuICAgKi9cbiAgZ2V0IHJlbGlhYmlsaXR5KCkge1xuICAgIHJldHVybiB0aGlzLl9kYXRhLnJlbGlhYmlsaXR5O1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdmVyc2lvbiB2YWx1ZS4gUG9zc2libGUgdmFsdWVzIGFyZSA0IGFuZCA1XG4gICAqIEByZXR1cm4ge1N0cmluZ30gYWNjb3VudCB2ZXJzaW9uIHZhbHVlXG4gICAqL1xuICBnZXQgdmVyc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy5fZGF0YS52ZXJzaW9uO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbG9hZHMgTWV0YVRyYWRlciBhY2NvdW50IGZyb20gQVBJXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gTWV0YVRyYWRlciBhY2NvdW50IGlzIHVwZGF0ZWRcbiAgICovXG4gIGFzeW5jIHJlbG9hZCgpIHtcbiAgICB0aGlzLl9kYXRhID0gYXdhaXQgdGhpcy5fbWV0YXRyYWRlckFjY291bnRDbGllbnQuZ2V0QWNjb3VudCh0aGlzLmlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIE1ldGFUcmFkZXIgYWNjb3VudC4gQ2xvdWQgYWNjb3VudCB0cmFuc2l0aW9ucyB0byBERUxFVElORyBzdGF0ZS4gXG4gICAqIEl0IHRha2VzIHNvbWUgdGltZSBmb3IgYW4gYWNjb3VudCB0byBiZSBldmVudHVhbGx5IGRlbGV0ZWQuIFNlbGYtaG9zdGVkIFxuICAgKiBhY2NvdW50IGlzIGRlbGV0ZWQgaW1tZWRpYXRlbHkuXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gYWNjb3VudCBpcyBzY2hlZHVsZWQgZm9yIGRlbGV0aW9uXG4gICAqL1xuICBhc3luYyByZW1vdmUoKSB7XG4gICAgdGhpcy5fY29ubmVjdGlvblJlZ2lzdHJ5LnJlbW92ZSh0aGlzLmlkKTtcbiAgICBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC5kZWxldGVBY2NvdW50KHRoaXMuaWQpO1xuICAgIGNvbnN0IGZpbGVNYW5hZ2VyID0gbmV3IEhpc3RvcnlGaWxlTWFuYWdlcih0aGlzLmlkKTtcbiAgICBhd2FpdCBmaWxlTWFuYWdlci5kZWxldGVTdG9yYWdlRnJvbURpc2soKTtcbiAgICBpZiAodGhpcy50eXBlICE9PSAnc2VsZi1ob3N0ZWQnKSB7XG4gICAgICB0cnkge1xuICAgICAgICBhd2FpdCB0aGlzLnJlbG9hZCgpO1xuICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgIGlmIChlcnIubmFtZSAhPT0gJ05vdEZvdW5kRXJyb3InKSB7XG4gICAgICAgICAgdGhyb3cgZXJyO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFNjaGVkdWxlcyBhY2NvdW50IGZvciBkZXBsb3ltZW50LiBJdCB0YWtlcyBzb21lIHRpbWUgZm9yIEFQSSBzZXJ2ZXIgdG8gYmUgc3RhcnRlZCBhbmQgYWNjb3VudCB0byByZWFjaCB0aGUgREVQTE9ZRURcbiAgICogc3RhdGVcbiAgICogQHJldHVybnMge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gYWNjb3VudCBpcyBzY2hlZHVsZWQgZm9yIGRlcGxveW1lbnRcbiAgICovXG4gIGFzeW5jIGRlcGxveSgpIHtcbiAgICBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC5kZXBsb3lBY2NvdW50KHRoaXMuaWQpO1xuICAgIGF3YWl0IHRoaXMucmVsb2FkKCk7XG4gIH1cblxuICAvKipcbiAgICogU2NoZWR1bGVzIGFjY291bnQgZm9yIHVuZGVwbG95bWVudC4gSXQgdGFrZXMgc29tZSB0aW1lIGZvciBBUEkgc2VydmVyIHRvIGJlIHN0b3BwZWQgYW5kIGFjY291bnQgdG8gcmVhY2ggdGhlXG4gICAqIFVOREVQTE9ZRUQgc3RhdGVcbiAgICogQHJldHVybnMge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gYWNjb3VudCBpcyBzY2hlZHVsZWQgZm9yIHVuZGVwbG95bWVudFxuICAgKi9cbiAgYXN5bmMgdW5kZXBsb3koKSB7XG4gICAgdGhpcy5fY29ubmVjdGlvblJlZ2lzdHJ5LnJlbW92ZSh0aGlzLmlkKTtcbiAgICBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC51bmRlcGxveUFjY291bnQodGhpcy5pZCk7XG4gICAgYXdhaXQgdGhpcy5yZWxvYWQoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTY2hlZHVsZXMgYWNjb3VudCBmb3IgcmVkZXBsb3ltZW50LiBJdCB0YWtlcyBzb21lIHRpbWUgZm9yIEFQSSBzZXJ2ZXIgdG8gYmUgcmVzdGFydGVkIGFuZCBhY2NvdW50IHRvIHJlYWNoIHRoZVxuICAgKiBERVBMT1lFRCBzdGF0ZVxuICAgKiBAcmV0dXJucyB7UHJvbWlzZX0gcHJvbWlzZSByZXNvbHZpbmcgd2hlbiBhY2NvdW50IGlzIHNjaGVkdWxlZCBmb3IgcmVkZXBsb3ltZW50XG4gICAqL1xuICBhc3luYyByZWRlcGxveSgpIHtcbiAgICBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC5yZWRlcGxveUFjY291bnQodGhpcy5pZCk7XG4gICAgYXdhaXQgdGhpcy5yZWxvYWQoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbmNyZWFzZXMgTWV0YVRyYWRlciBhY2NvdW50IHJlbGlhYmlsaXR5LiBUaGUgYWNjb3VudCB3aWxsIGJlIHRlbXBvcmFyeSBzdG9wcGVkIHRvIHBlcmZvcm0gdGhpcyBhY3Rpb25cbiAgICogQHJldHVybnMge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gYWNjb3VudCByZWxpYWJpbGl0eSBpcyBpbmNyZWFzZWRcbiAgICovXG4gIGFzeW5jIGluY3JlYXNlUmVsaWFiaWxpdHkoKSB7XG4gICAgYXdhaXQgdGhpcy5fbWV0YXRyYWRlckFjY291bnRDbGllbnQuaW5jcmVhc2VSZWxpYWJpbGl0eSh0aGlzLmlkKTtcbiAgICBhd2FpdCB0aGlzLnJlbG9hZCgpO1xuICB9XG5cbiAgLyoqXG4gICAqIFdhaXRzIHVudGlsIEFQSSBzZXJ2ZXIgaGFzIGZpbmlzaGVkIGRlcGxveW1lbnQgYW5kIGFjY291bnQgcmVhY2hlZCB0aGUgREVQTE9ZRUQgc3RhdGVcbiAgICogQHBhcmFtIHtOdW1iZXJ9IHRpbWVvdXRJblNlY29uZHMgd2FpdCB0aW1lb3V0IGluIHNlY29uZHMsIGRlZmF1bHQgaXMgNW1cbiAgICogQHBhcmFtIHtOdW1iZXJ9IGludGVydmFsSW5NaWxsaXNlY29uZHMgaW50ZXJ2YWwgYmV0d2VlbiBhY2NvdW50IHJlbG9hZHMgd2hpbGUgd2FpdGluZyBmb3IgYSBjaGFuZ2UsIGRlZmF1bHQgaXMgMXNcbiAgICogQHJldHVybiB7UHJvbWlzZX0gcHJvbWlzZSB3aGljaCByZXNvbHZlcyB3aGVuIGFjY291bnQgaXMgZGVwbG95ZWRcbiAgICogQHRocm93cyB7VGltZW91dEVycm9yfSBpZiBhY2NvdW50IGhhdmUgbm90IHJlYWNoZWQgdGhlIERFUExPWUVEIHN0YXRlIHdpdGhpbmcgdGltZW91dCBhbGxvd2VkXG4gICAqL1xuICBhc3luYyB3YWl0RGVwbG95ZWQodGltZW91dEluU2Vjb25kcyA9IDMwMCwgaW50ZXJ2YWxJbk1pbGxpc2Vjb25kcyA9IDEwMDApIHtcbiAgICBsZXQgc3RhcnRUaW1lID0gRGF0ZS5ub3coKTtcbiAgICBhd2FpdCB0aGlzLnJlbG9hZCgpO1xuICAgIHdoaWxlICh0aGlzLnN0YXRlICE9PSAnREVQTE9ZRUQnICYmIChzdGFydFRpbWUgKyB0aW1lb3V0SW5TZWNvbmRzICogMTAwMCkgPiBEYXRlLm5vdygpKSB7XG4gICAgICBhd2FpdCB0aGlzLl9kZWxheShpbnRlcnZhbEluTWlsbGlzZWNvbmRzKTtcbiAgICAgIGF3YWl0IHRoaXMucmVsb2FkKCk7XG4gICAgfVxuICAgIGlmICh0aGlzLnN0YXRlICE9PSAnREVQTE9ZRUQnKSB7XG4gICAgICB0aHJvdyBuZXcgVGltZW91dEVycm9yKCdUaW1lZCBvdXQgd2FpdGluZyBmb3IgYWNjb3VudCAnICsgdGhpcy5pZCArICcgdG8gYmUgZGVwbG95ZWQnKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogV2FpdHMgdW50aWwgQVBJIHNlcnZlciBoYXMgZmluaXNoZWQgdW5kZXBsb3ltZW50IGFuZCBhY2NvdW50IHJlYWNoZWQgdGhlIFVOREVQTE9ZRUQgc3RhdGVcbiAgICogQHBhcmFtIHtOdW1iZXJ9IHRpbWVvdXRJblNlY29uZHMgd2FpdCB0aW1lb3V0IGluIHNlY29uZHMsIGRlZmF1bHQgaXMgNW1cbiAgICogQHBhcmFtIHtOdW1iZXJ9IGludGVydmFsSW5NaWxsaXNlY29uZHMgaW50ZXJ2YWwgYmV0d2VlbiBhY2NvdW50IHJlbG9hZHMgd2hpbGUgd2FpdGluZyBmb3IgYSBjaGFuZ2UsIGRlZmF1bHQgaXMgMXNcbiAgICogQHJldHVybiB7UHJvbWlzZX0gcHJvbWlzZSB3aGljaCByZXNvbHZlcyB3aGVuIGFjY291bnQgaXMgZGVwbG95ZWRcbiAgICogQHRocm93cyB7VGltZW91dEVycm9yfSBpZiBhY2NvdW50IGhhdmUgbm90IHJlYWNoZWQgdGhlIFVOREVQTE9ZRUQgc3RhdGUgd2l0aGluZyB0aW1lb3V0IGFsbG93ZWRcbiAgICovXG4gIGFzeW5jIHdhaXRVbmRlcGxveWVkKHRpbWVvdXRJblNlY29uZHMgPSAzMDAsIGludGVydmFsSW5NaWxsaXNlY29uZHMgPSAxMDAwKSB7XG4gICAgbGV0IHN0YXJ0VGltZSA9IERhdGUubm93KCk7XG4gICAgYXdhaXQgdGhpcy5yZWxvYWQoKTtcbiAgICB3aGlsZSAodGhpcy5zdGF0ZSAhPT0gJ1VOREVQTE9ZRUQnICYmIChzdGFydFRpbWUgKyB0aW1lb3V0SW5TZWNvbmRzICogMTAwMCkgPiBEYXRlLm5vdygpKSB7XG4gICAgICBhd2FpdCB0aGlzLl9kZWxheShpbnRlcnZhbEluTWlsbGlzZWNvbmRzKTtcbiAgICAgIGF3YWl0IHRoaXMucmVsb2FkKCk7XG4gICAgfVxuICAgIGlmICh0aGlzLnN0YXRlICE9PSAnVU5ERVBMT1lFRCcpIHtcbiAgICAgIHRocm93IG5ldyBUaW1lb3V0RXJyb3IoJ1RpbWVkIG91dCB3YWl0aW5nIGZvciBhY2NvdW50ICcgKyB0aGlzLmlkICsgJyB0byBiZSB1bmRlcGxveWVkJyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFdhaXRzIHVudGlsIGFjY291bnQgaGFzIGJlZW4gZGVsZXRlZFxuICAgKiBAcGFyYW0ge051bWJlcn0gdGltZW91dEluU2Vjb25kcyB3YWl0IHRpbWVvdXQgaW4gc2Vjb25kcywgZGVmYXVsdCBpcyA1bVxuICAgKiBAcGFyYW0ge051bWJlcn0gaW50ZXJ2YWxJbk1pbGxpc2Vjb25kcyBpbnRlcnZhbCBiZXR3ZWVuIGFjY291bnQgcmVsb2FkcyB3aGlsZSB3YWl0aW5nIGZvciBhIGNoYW5nZSwgZGVmYXVsdCBpcyAxc1xuICAgKiBAcmV0dXJuIHtQcm9taXNlfSBwcm9taXNlIHdoaWNoIHJlc29sdmVzIHdoZW4gYWNjb3VudCBpcyBkZWxldGVkXG4gICAqIEB0aHJvd3Mge1RpbWVvdXRFcnJvcn0gaWYgYWNjb3VudCB3YXMgbm90IGRlbGV0ZWQgd2l0aGluZyB0aW1lb3V0IGFsbG93ZWRcbiAgICovXG4gIGFzeW5jIHdhaXRSZW1vdmVkKHRpbWVvdXRJblNlY29uZHMgPSAzMDAsIGludGVydmFsSW5NaWxsaXNlY29uZHMgPSAxMDAwKSB7XG4gICAgbGV0IHN0YXJ0VGltZSA9IERhdGUubm93KCk7XG4gICAgdHJ5IHtcbiAgICAgIGF3YWl0IHRoaXMucmVsb2FkKCk7XG4gICAgICB3aGlsZSAoc3RhcnRUaW1lICsgdGltZW91dEluU2Vjb25kcyAqIDEwMDAgPiBEYXRlLm5vdygpKSB7XG4gICAgICAgIGF3YWl0IHRoaXMuX2RlbGF5KGludGVydmFsSW5NaWxsaXNlY29uZHMpO1xuICAgICAgICBhd2FpdCB0aGlzLnJlbG9hZCgpO1xuICAgICAgfVxuICAgICAgdGhyb3cgbmV3IFRpbWVvdXRFcnJvcignVGltZWQgb3V0IHdhaXRpbmcgZm9yIGFjY291bnQgJyArIHRoaXMuaWQgKyAnIHRvIGJlIGRlbGV0ZWQnKTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGlmIChlcnIubmFtZSA9PT0gJ05vdEZvdW5kRXJyb3InKSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IGVycjtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogV2FpdHMgdW50aWwgQVBJIHNlcnZlciBoYXMgY29ubmVjdGVkIHRvIHRoZSB0ZXJtaW5hbCBhbmQgdGVybWluYWwgaGFzIGNvbm5lY3RlZCB0byB0aGUgYnJva2VyXG4gICAqIEBwYXJhbSB7TnVtYmVyfSB0aW1lb3V0SW5TZWNvbmRzIHdhaXQgdGltZW91dCBpbiBzZWNvbmRzLCBkZWZhdWx0IGlzIDVtXG4gICAqIEBwYXJhbSB7TnVtYmVyfSBpbnRlcnZhbEluTWlsbGlzZWNvbmRzIGludGVydmFsIGJldHdlZW4gYWNjb3VudCByZWxvYWRzIHdoaWxlIHdhaXRpbmcgZm9yIGEgY2hhbmdlLCBkZWZhdWx0IGlzIDFzXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2Ugd2hpY2ggcmVzb2x2ZXMgd2hlbiBBUEkgc2VydmVyIGlzIGNvbm5lY3RlZCB0byB0aGUgYnJva2VyXG4gICAqIEB0aHJvd3Mge1RpbWVvdXRFcnJvcn0gaWYgYWNjb3VudCBoYXZlIG5vdCBjb25uZWN0ZWQgdG8gdGhlIGJyb2tlciB3aXRoaW5nIHRpbWVvdXQgYWxsb3dlZFxuICAgKi9cbiAgYXN5bmMgd2FpdENvbm5lY3RlZCh0aW1lb3V0SW5TZWNvbmRzID0gMzAwLCBpbnRlcnZhbEluTWlsbGlzZWNvbmRzID0gMTAwMCkge1xuICAgIGxldCBzdGFydFRpbWUgPSBEYXRlLm5vdygpO1xuICAgIGF3YWl0IHRoaXMucmVsb2FkKCk7XG4gICAgd2hpbGUgKHRoaXMuY29ubmVjdGlvblN0YXR1cyAhPT0gJ0NPTk5FQ1RFRCcgJiYgKHN0YXJ0VGltZSArIHRpbWVvdXRJblNlY29uZHMgKiAxMDAwKSA+IERhdGUubm93KCkpIHtcbiAgICAgIGF3YWl0IHRoaXMuX2RlbGF5KGludGVydmFsSW5NaWxsaXNlY29uZHMpO1xuICAgICAgYXdhaXQgdGhpcy5yZWxvYWQoKTtcbiAgICB9XG4gICAgaWYgKHRoaXMuY29ubmVjdGlvblN0YXR1cyAhPT0gJ0NPTk5FQ1RFRCcpIHtcbiAgICAgIHRocm93IG5ldyBUaW1lb3V0RXJyb3IoJ1RpbWVkIG91dCB3YWl0aW5nIGZvciBhY2NvdW50ICcgKyB0aGlzLmlkICsgJyB0byBjb25uZWN0IHRvIHRoZSBicm9rZXInKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ29ubmVjdHMgdG8gTWV0YUFwaS4gVGhlcmUgaXMgb25seSBvbmUgY29ubmVjdGlvbiBwZXIgYWNjb3VudC4gU3Vic2VxdWVudCBjYWxscyB0byB0aGlzIG1ldGhvZCB3aWxsIHJldHVybiB0aGUgc2FtZSBjb25uZWN0aW9uLlxuICAgKiBAcGFyYW0ge0hpc3RvcnlTdG9yYWdlfSBoaXN0b3J5U3RvcmFnZSBvcHRpb25hbCBoaXN0b3J5IHN0b3JhZ2VcbiAgICogQHBhcmFtIHtEYXRlfSBbaGlzdG9yeVN0YXJ0VGltZV0gaGlzdG9yeSBzdGFydCB0aW1lLiBVc2VkIGZvciB0ZXN0c1xuICAgKiBAcmV0dXJuIHtTdHJlYW1pbmdNZXRhQXBpQ29ubmVjdGlvbn0gTWV0YUFwaSBjb25uZWN0aW9uXG4gICAqL1xuICBnZXRTdHJlYW1pbmdDb25uZWN0aW9uKGhpc3RvcnlTdG9yYWdlLCBoaXN0b3J5U3RhcnRUaW1lKSB7XG4gICAgcmV0dXJuIHRoaXMuX2Nvbm5lY3Rpb25SZWdpc3RyeS5jb25uZWN0KHRoaXMsIGhpc3RvcnlTdG9yYWdlLCBoaXN0b3J5U3RhcnRUaW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25uZWN0cyB0byBNZXRhQXBpIHZpYSBSUEMgY29ubmVjdGlvbi5cbiAgICogQHJldHVybnMge1JwY01ldGFBcGlDb25uZWN0aW9ufSBNZXRhQXBpIGNvbm5lY3Rpb25cbiAgICovXG4gIGdldFJQQ0Nvbm5lY3Rpb24oKSB7XG4gICAgcmV0dXJuIG5ldyBScGNNZXRhQXBpQ29ubmVjdGlvbih0aGlzLl9tZXRhQXBpV2Vic29ja2V0Q2xpZW50LCB0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIE1ldGFUcmFkZXIgYWNjb3VudCBkYXRhXG4gICAqIEBwYXJhbSB7TWV0YXRyYWRlckFjY291bnRVcGRhdGVEdG99IGFjY291bnQgTWV0YVRyYWRlciBhY2NvdW50IHVwZGF0ZVxuICAgKiBAcmV0dXJuIHtQcm9taXNlfSBwcm9taXNlIHJlc29sdmluZyB3aGVuIGFjY291bnQgaXMgdXBkYXRlZFxuICAgKi9cbiAgYXN5bmMgdXBkYXRlKGFjY291bnQpIHtcbiAgICBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC51cGRhdGVBY2NvdW50KHRoaXMuaWQsIGFjY291bnQpO1xuICAgIGF3YWl0IHRoaXMucmVsb2FkKCk7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGV4cGVydCBhZHZpc29yIG9mIGN1cnJlbnQgYWNjb3VudFxuICAgKiBAcmV0dXJucyB7UHJvbWlzZTxFeHBlcnRBZHZpc29yW10+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIGFuIGFycmF5IG9mIGV4cGVydCBhZHZpc29yIGVudGl0aWVzXG4gICAqL1xuICBhc3luYyBnZXRFeHBlcnRBZHZpc29ycygpIHtcbiAgICB0aGlzLl9jaGVja0V4cGVydEFkdmlzb3JBbGxvd2VkKCk7XG4gICAgbGV0IGV4cGVydEFkdmlzb3JzID0gYXdhaXQgdGhpcy5fZXhwZXJ0QWR2aXNvckNsaWVudC5nZXRFeHBlcnRBZHZpc29ycyh0aGlzLmlkKTtcbiAgICByZXR1cm4gZXhwZXJ0QWR2aXNvcnMubWFwKGUgPT4gbmV3IEV4cGVydEFkdmlzb3IoZSwgdGhpcy5pZCwgdGhpcy5fZXhwZXJ0QWR2aXNvckNsaWVudCkpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhIGV4cGVydCBhZHZpc29yIG9mIGN1cnJlbnQgYWNjb3VudCBieSBpZFxuICAgKiBAcGFyYW0ge1N0cmluZ30gZXhwZXJ0SWQgZXhwZXJ0IGFkdmlzb3IgaWRcbiAgICogQHJldHVybnMge1Byb21pc2U8RXhwZXJ0QWR2aXNvcj59IHByb21pc2UgcmVzb2x2aW5nIHdpdGggZXhwZXJ0IGFkdmlzb3IgZW50aXR5XG4gICAqL1xuICBhc3luYyBnZXRFeHBlcnRBZHZpc29yKGV4cGVydElkKSB7XG4gICAgdGhpcy5fY2hlY2tFeHBlcnRBZHZpc29yQWxsb3dlZCgpO1xuICAgIGxldCBleHBlcnRBZHZpc29yID0gYXdhaXQgdGhpcy5fZXhwZXJ0QWR2aXNvckNsaWVudC5nZXRFeHBlcnRBZHZpc29yKHRoaXMuaWQsIGV4cGVydElkKTtcbiAgICByZXR1cm4gbmV3IEV4cGVydEFkdmlzb3IoZXhwZXJ0QWR2aXNvciwgdGhpcy5pZCwgdGhpcy5fZXhwZXJ0QWR2aXNvckNsaWVudCk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBleHBlcnQgYWR2aXNvclxuICAgKiBAcGFyYW0ge1N0cmluZ30gZXhwZXJ0SWQgZXhwZXJ0IGFkdmlzb3IgaWRcbiAgICogQHBhcmFtIHtOZXdFeHBlcnRBZHZpc29yRHRvfSBleHBlcnQgZXhwZXJ0IGFkdmlzb3IgZGF0YVxuICAgKiBAcmV0dXJucyB7UHJvbWlzZTxFeHBlcnRBZHZpc29yPn0gcHJvbWlzZSByZXNvbHZpbmcgd2l0aCBleHBlcnQgYWR2aXNvciBlbnRpdHlcbiAgICovXG4gIGFzeW5jIGNyZWF0ZUV4cGVydEFkdmlzb3IoZXhwZXJ0SWQsIGV4cGVydCkge1xuICAgIHRoaXMuX2NoZWNrRXhwZXJ0QWR2aXNvckFsbG93ZWQoKTtcbiAgICBhd2FpdCB0aGlzLl9leHBlcnRBZHZpc29yQ2xpZW50LnVwZGF0ZUV4cGVydEFkdmlzb3IodGhpcy5pZCwgZXhwZXJ0SWQsIGV4cGVydCk7XG4gICAgcmV0dXJuIHRoaXMuZ2V0RXhwZXJ0QWR2aXNvcihleHBlcnRJZCk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBoaXN0b3JpY2FsIGNhbmRsZXMgZm9yIGEgc3BlY2lmaWMgc3ltYm9sIGFuZCB0aW1lZnJhbWUgZnJvbSB0aGUgTWV0YVRyYWRlciBhY2NvdW50LlxuICAgKiBTZWUgaHR0cHM6Ly9tZXRhYXBpLmNsb3VkL2RvY3MvY2xpZW50L3Jlc3RBcGkvYXBpL3JldHJpZXZlTWFya2V0RGF0YS9yZWFkSGlzdG9yaWNhbENhbmRsZXMvXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBzeW1ib2wgc3ltYm9sIHRvIHJldHJpZXZlIGNhbmRsZXMgZm9yIChlLmcuIGEgY3VycmVuY3kgcGFpciBvciBhbiBpbmRleClcbiAgICogQHBhcmFtIHtzdHJpbmd9IHRpbWVmcmFtZSBkZWZpbmVzIHRoZSB0aW1lZnJhbWUgYWNjb3JkaW5nIHRvIHdoaWNoIHRoZSBjYW5kbGVzIG11c3QgYmUgZ2VuZXJhdGVkLiBBbGxvd2VkIHZhbHVlc1xuICAgKiBmb3IgTVQ1IGFyZSAxbSwgMm0sIDNtLCA0bSwgNW0sIDZtLCAxMG0sIDEybSwgMTVtLCAyMG0sIDMwbSwgMWgsIDJoLCAzaCwgNGgsIDZoLCA4aCwgMTJoLCAxZCwgMXcsIDFtbi4gQWxsb3dlZFxuICAgKiB2YWx1ZXMgZm9yIE1UNCBhcmUgMW0sIDVtLCAxNW0gMzBtLCAxaCwgNGgsIDFkLCAxdywgMW1uXG4gICAqIEBwYXJhbSB7RGF0ZX0gW3N0YXJ0VGltZV0gdGltZSB0byBzdGFydCBsb2FkaW5nIGNhbmRsZXMgZnJvbS4gTm90ZSB0aGF0IGNhbmRsZXMgYXJlIGxvYWRlZCBpbiBiYWNrd2FyZHMgZGlyZWN0aW9uLCBzb1xuICAgKiB0aGlzIHNob3VsZCBiZSB0aGUgbGF0ZXN0IHRpbWUuIExlYXZlIGVtcHR5IHRvIHJlcXVlc3QgbGF0ZXN0IGNhbmRsZXMuXG4gICAqIEBwYXJhbSB7bnVtYmVyfSBbbGltaXRdIG1heGltdW0gbnVtYmVyIG9mIGNhbmRsZXMgdG8gcmV0cmlldmUuIE11c3QgYmUgbGVzcyBvciBlcXVhbCB0byAxMDAwXG4gICAqIEByZXR1cm4ge1Byb21pc2U8QXJyYXk8TWV0YXRyYWRlckNhbmRsZT4+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIGhpc3RvcmljYWwgY2FuZGxlcyBkb3dubG9hZGVkXG4gICAqL1xuICBnZXRIaXN0b3JpY2FsQ2FuZGxlcyhzeW1ib2wsIHRpbWVmcmFtZSwgc3RhcnRUaW1lLCBsaW1pdCkge1xuICAgIHJldHVybiB0aGlzLl9oaXN0b3JpY2FsTWFya2V0RGF0YUNsaWVudC5nZXRIaXN0b3JpY2FsQ2FuZGxlcyh0aGlzLmlkLCBzeW1ib2wsIHRpbWVmcmFtZSwgc3RhcnRUaW1lLCBsaW1pdCk7XG4gIH1cbiAgXG4gIC8qKlxuICAgKiBSZXR1cm5zIGhpc3RvcmljYWwgdGlja3MgZm9yIGEgc3BlY2lmaWMgc3ltYm9sIGZyb20gdGhlIE1ldGFUcmFkZXIgYWNjb3VudC4gVGhpcyBBUEkgaXMgbm90IHN1cHBvcnRlZCBieSBNVDRcbiAgICogYWNjb3VudHMuXG4gICAqIFNlZSBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9jbGllbnQvcmVzdEFwaS9hcGkvcmV0cmlldmVNYXJrZXREYXRhL3JlYWRIaXN0b3JpY2FsVGlja3MvXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBzeW1ib2wgc3ltYm9sIHRvIHJldHJpZXZlIHRpY2tzIGZvciAoZS5nLiBhIGN1cnJlbmN5IHBhaXIgb3IgYW4gaW5kZXgpXG4gICAqIEBwYXJhbSB7RGF0ZX0gW3N0YXJ0VGltZV0gdGltZSB0byBzdGFydCBsb2FkaW5nIHRpY2tzIGZyb20uIE5vdGUgdGhhdCBjYW5kbGVzIGFyZSBsb2FkZWQgaW4gZm9yd2FyZCBkaXJlY3Rpb24sIHNvXG4gICAqIHRoaXMgc2hvdWxkIGJlIHRoZSBlYXJsaWVzdCB0aW1lLiBMZWF2ZSBlbXB0eSB0byByZXF1ZXN0IGxhdGVzdCBjYW5kbGVzLlxuICAgKiBAcGFyYW0ge251bWJlcn0gW29mZnNldF0gbnVtYmVyIG9mIHRpY2tzIHRvIHNraXAgKHlvdSBjYW4gdXNlIGl0IHRvIGF2b2lkIHJlcXVlc3RpbmcgdGlja3MgZnJvbSBwcmV2aW91cyByZXF1ZXN0XG4gICAqIHR3aWNlKVxuICAgKiBAcGFyYW0ge251bWJlcn0gW2xpbWl0XSBtYXhpbXVtIG51bWJlciBvZiB0aWNrcyB0byByZXRyaWV2ZS4gTXVzdCBiZSBsZXNzIG9yIGVxdWFsIHRvIDEwMDBcbiAgICogQHJldHVybiB7UHJvbWlzZTxBcnJheTxNZXRhdHJhZGVyVGljaz4+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIGhpc3RvcmljYWwgdGlja3MgZG93bmxvYWRlZFxuICAgKi9cbiAgZ2V0SGlzdG9yaWNhbFRpY2tzKHN5bWJvbCwgc3RhcnRUaW1lLCBvZmZzZXQsIGxpbWl0KSB7XG4gICAgcmV0dXJuIHRoaXMuX2hpc3RvcmljYWxNYXJrZXREYXRhQ2xpZW50LmdldEhpc3RvcmljYWxUaWNrcyh0aGlzLmlkLCBzeW1ib2wsIHN0YXJ0VGltZSwgb2Zmc2V0LCBsaW1pdCk7XG4gIH1cblxuICBfY2hlY2tFeHBlcnRBZHZpc29yQWxsb3dlZCgpIHtcbiAgICBpZiAodGhpcy52ZXJzaW9uICE9PSA0IHx8IHRoaXMudHlwZSAhPT0gJ2Nsb3VkLWcxJykge1xuICAgICAgdGhyb3cgbmV3IFZhbGlkYXRpb25FcnJvcignQ3VzdG9tIGV4cGVydCBhZHZpc29yIGlzIGF2YWlsYWJsZSBvbmx5IGZvciBNVDQgRzEgYWNjb3VudHMnKTtcbiAgICB9XG4gIH1cblxuICBfZGVsYXkodGltZW91dEluTWlsbGlzZWNvbmRzKSB7XG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKHJlcyA9PiBzZXRUaW1lb3V0KHJlcywgdGltZW91dEluTWlsbGlzZWNvbmRzKSk7XG4gIH1cblxufVxuIl19