'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _httpClient = require('../clients/httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _provisioningProfile = require('../clients/metaApi/provisioningProfile.client');

var _provisioningProfile2 = _interopRequireDefault(_provisioningProfile);

var _provisioningProfileApi = require('./provisioningProfileApi');

var _provisioningProfileApi2 = _interopRequireDefault(_provisioningProfileApi);

var _metaApiWebsocket = require('../clients/metaApi/metaApiWebsocket.client');

var _metaApiWebsocket2 = _interopRequireDefault(_metaApiWebsocket);

var _metatraderAccountApi = require('./metatraderAccountApi');

var _metatraderAccountApi2 = _interopRequireDefault(_metatraderAccountApi);

var _metatraderAccount = require('../clients/metaApi/metatraderAccount.client');

var _metatraderAccount2 = _interopRequireDefault(_metatraderAccount);

var _metatraderDemoAccountApi = require('./metatraderDemoAccountApi');

var _metatraderDemoAccountApi2 = _interopRequireDefault(_metatraderDemoAccountApi);

var _metatraderDemoAccount = require('../clients/metaApi/metatraderDemoAccount.client');

var _metatraderDemoAccount2 = _interopRequireDefault(_metatraderDemoAccount);

var _historicalMarketData = require('../clients/metaApi/historicalMarketData.client');

var _historicalMarketData2 = _interopRequireDefault(_historicalMarketData);

var _connectionRegistry = require('./connectionRegistry');

var _connectionRegistry2 = _interopRequireDefault(_connectionRegistry);

var _errorHandler = require('../clients/errorHandler');

var _optionsValidator = require('../clients/optionsValidator');

var _optionsValidator2 = _interopRequireDefault(_optionsValidator);

var _latencyMonitor = require('./latencyMonitor');

var _latencyMonitor2 = _interopRequireDefault(_latencyMonitor);

var _expertAdvisor = require('../clients/metaApi/expertAdvisor.client');

var _expertAdvisor2 = _interopRequireDefault(_expertAdvisor);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Request retry options
 * @typedef {Object} RetryOpts
 * @property {Number} [retries] maximum amount of request retries, default value is 5
 * @property {Number} [minDelayInSeconds] minimum delay in seconds until request retry, default value is 1
 * @property {Number} [maxDelayInSeconds] maximum delay in seconds until request retry, default value is 30
 */

/**
 * Subscriptions refresh options
 * @typedef {Object} RefreshSubscriptionsOpts
 * @property {Number} [minDelayInSeconds] minimum delay in seconds until subscriptions refresh request,
 * default value is 1
 * @property {Number} [maxDelayInSeconds] maximum delay in seconds until subscriptions refresh request,
 * default value is 600
 */

/**
 * MetaApi options
 * @typedef {Object} MetaApiOpts
 * @property {String} [application] application id
 * @property {String} [domain] domain to connect to, default is agiliumtrade.agiliumtrade.ai
 * @property {String} [region] region to connect to
 * @property {Number} [requestTimeout] timeout for socket requests in seconds
 * @property {Number} [connectTimeout] timeout for connecting to server in seconds
 * @property {Number} [packetOrderingTimeout] packet ordering timeout in seconds
 * @property {PacketLoggerOpts} [packetLogger] packet logger options
 * @property {Boolean} [enableLatencyMonitor] an option to enable latency tracking
 * @property {Boolean} [enableLatencyTracking] an option to enable latency tracking
 * @property {SynchronizationThrottlerOpts} [synchronizationThrottler] options for synchronization throttler
 * @property {RetryOpts} [retryOpts] options for request retries
 * @property {Boolean} [useSharedClientApi] option to use a shared server
 * @property {RefreshSubscriptionsOpts} [refreshSubscriptionsOpts] subscriptions refresh options
 * @property {Number} [unsubscribeThrottlingIntervalInSeconds] a timeout in seconds for throttling repeat unsubscribe
 * requests when synchronization packets still arrive after unsubscription, default is 10 seconds
 */

/**
 * MetaApi MetaTrader API SDK
 */
class MetaApi {

  /**
   * Enables using Log4js logger with extended log levels for debugging instead of
   * console.* functions. Note that log4js configuration performed by the user.
   */
  static enableLog4jsLogging() {
    _logger2.default.useLog4js();
  }

  /**
   * Constructs MetaApi class instance
   * @param {String} token authorization token
   * @param {MetaApiOpts} opts application options
   */
  // eslint-disable-next-line complexity
  constructor(token, opts) {
    const validator = new _optionsValidator2.default();
    opts = opts || {};
    const application = opts.application || 'MetaApi';
    const domain = opts.domain || 'agiliumtrade.agiliumtrade.ai';
    const requestTimeout = validator.validateNonZero(opts.requestTimeout, 60, 'requestTimeout');
    const historicalMarketDataRequestTimeout = validator.validateNonZero(opts.historicalMarketDataRequestTimeout, 240, 'historicalMarketDataRequestTimeout');
    const connectTimeout = validator.validateNonZero(opts.connectTimeout, 60, 'connectTimeout');
    const packetOrderingTimeout = validator.validateNonZero(opts.packetOrderingTimeout, 60, 'packetOrderingTimeout');
    const retryOpts = opts.retryOpts || {};
    const packetLogger = opts.packetLogger || {};
    const synchronizationThrottler = opts.synchronizationThrottler || {};
    const demoAccountRequestTimeout = validator.validateNonZero(opts.demoAccountRequestTimeout, 240, 'demoAccountRequestTimeout');
    if (!application.match(/[a-zA-Z0-9_]+/)) {
      throw new _errorHandler.ValidationError('Application name must be non-empty string consisting from letters, digits and _ only');
    }
    const useSharedClientApi = opts.useSharedClientApi || false;
    const refreshSubscriptionsOpts = opts.refreshSubscriptionsOpts || {};
    let httpClient = new _httpClient2.default(requestTimeout, retryOpts);
    let historicalMarketDataHttpClient = new _httpClient2.default(historicalMarketDataRequestTimeout, retryOpts);
    let demoAccountHttpClient = new _httpClient2.default(demoAccountRequestTimeout, retryOpts);
    this._metaApiWebsocketClient = new _metaApiWebsocket2.default(httpClient, token, { application, domain, requestTimeout,
      connectTimeout, packetLogger, packetOrderingTimeout, synchronizationThrottler, retryOpts, useSharedClientApi,
      region: opts.region, unsubscribeThrottlingIntervalInSeconds: opts.unsubscribeThrottlingIntervalInSeconds });
    this._provisioningProfileApi = new _provisioningProfileApi2.default(new _provisioningProfile2.default(httpClient, token, domain));
    this._connectionRegistry = new _connectionRegistry2.default(this._metaApiWebsocketClient, application, refreshSubscriptionsOpts);
    let historicalMarketDataClient = new _historicalMarketData2.default(historicalMarketDataHttpClient, token, domain);
    this._metatraderAccountApi = new _metatraderAccountApi2.default(new _metatraderAccount2.default(httpClient, token, domain), this._metaApiWebsocketClient, this._connectionRegistry, new _expertAdvisor2.default(httpClient, token, domain), historicalMarketDataClient);
    this._metatraderDemoAccountApi = new _metatraderDemoAccountApi2.default(new _metatraderDemoAccount2.default(demoAccountHttpClient, token, domain));
    if (opts.enableLatencyTracking || opts.enableLatencyMonitor) {
      this._latencyMonitor = new _latencyMonitor2.default();
      this._metaApiWebsocketClient.addLatencyListener(this._latencyMonitor);
    }
  }

  /**
   * Returns provisioning profile API
   * @returns {ProvisioningProfileApi} provisioning profile API
   */
  get provisioningProfileApi() {
    return this._provisioningProfileApi;
  }

  /**
   * Returns MetaTrader account API
   * @return {MetatraderAccountApi} MetaTrader account API
   */
  get metatraderAccountApi() {
    return this._metatraderAccountApi;
  }

  /**
   * Returns MetaTrader demo account API
   * @return {MetatraderDemoAccountApi} MetaTrader demo account API
   */
  get metatraderDemoAccountApi() {
    return this._metatraderDemoAccountApi;
  }

  /**
   * Returns MetaApi application latency monitor
   * @return {LatencyMonitor} latency monitor
   */
  get latencyMonitor() {
    return this._latencyMonitor;
  }

  /**
   * Closes all clients and connections
   */
  close() {
    this._metaApiWebsocketClient.removeLatencyListener(this._latencyMonitor);
    this._metaApiWebsocketClient.close();
  }

}
exports.default = MetaApi;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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