'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _synchronizationListener = require('../clients/metaApi/synchronizationListener');

var _synchronizationListener2 = _interopRequireDefault(_synchronizationListener);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Abstract class which defines MetaTrader history storage interface.
 */
class HistoryStorage extends _synchronizationListener2.default {

  /**
   * Constructs the history storage
   */
  constructor() {
    super();
    this._orderSynchronizationFinished = {};
    this._dealSynchronizationFinished = {};
  }

  /**
   * Initializes the storage and loads required data from a persistent storage
   */
  async initialize() {}

  /**
   * Returns flag indicating whether order history synchronization have finished
   * @return {Boolean} flag indicating whether order history synchronization have finished
   */
  get orderSynchronizationFinished() {
    return (0, _values2.default)(this._orderSynchronizationFinished).reduce((acc, r) => acc || r, false);
  }

  /**
   * Returns flag indicating whether deal history synchronization have finished
   * @return {Boolean} flag indicating whether deal history synchronization have finished
   */
  get dealSynchronizationFinished() {
    return (0, _values2.default)(this._dealSynchronizationFinished).reduce((acc, r) => acc || r, false);
  }

  /**
   * Clears the storage and deletes persistent data
   */
  async clear() {
    throw Error('Abstract method clear has no implementation');
  }

  /**
   * Returns the time of the last history order record stored in the history storage
   * @param {String} [instanceIndex] index of an account instance connected
   * @returns {Date} the time of the last history order record stored in the history storage
   */
  async lastHistoryOrderTime(instanceIndex) {
    throw Error('Abstract method lastHistoryOrderTime has no implementation');
  }

  /**
   * Returns the time of the last history deal record stored in the history storage
   * @param {String} [instanceIndex] index of an account instance connected
   * @returns {Date} the time of the last history deal record stored in the history storage
   */
  async lastDealTime(instanceIndex) {
    throw Error('Abstract method lastDealTime has no implementation');
  }

  /**
   * Invoked when a new MetaTrader history order is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrderAdded(instanceIndex, historyOrder) {
    throw Error('Abstract method onHistoryOrderAdded has no implementation');
  }

  /**
   * Invoked when a new MetaTrader history deal is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderDeal} deal new MetaTrader history deal
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealAdded(instanceIndex, deal) {
    throw Error('Abstract method onDealAdded has no implementation');
  }

  /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealsSynchronized(instanceIndex, synchronizationId) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._dealSynchronizationFinished['' + instance] = true;
  }

  /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrdersSynchronized(instanceIndex, synchronizationId) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._orderSynchronizationFinished['' + instance] = true;
  }

  /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   */
  onConnected(instanceIndex) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._orderSynchronizationFinished['' + instance] = false;
    this._dealSynchronizationFinished['' + instance] = false;
  }

}
exports.default = HistoryStorage;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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