'use strict';

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _fileManager = require('./fileManager');

var _fileManager2 = _interopRequireDefault(_fileManager);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _util = require('util');

var _logger = require('../../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * History storage file manager which saves and loads history on disk
 */
module.exports = class HistoryFileManager extends _fileManager2.default {

  /**
   * Constructs the history file manager instance
   * @param {String} accountId MetaApi account id
   * @param {String} application MetaApi application id
   * @param {HistoryStorage} historyStorage history storage
   */
  constructor(accountId, application, historyStorage) {
    super(accountId, application, historyStorage);
    this._dealsSize = [];
    this._startNewDealIndex = -1;
    this._historyOrdersSize = [];
    this._startNewOrderIndex = -1;
    this._isUpdating = false;
    this._logger = _logger2.default.getLogger('HistoryFileManager');
  }

  /**
   * Helper function to calculate object size in bytes in utf-8 encoding
   * @returns {number} size of object in bytes
   */
  getItemSize(item) {
    return new _util.TextEncoder().encode((0, _stringify2.default)(item)).length;
  }

  /**
   * Retrieves history from saved file
   * @returns {Object} object with deals and historyOrders
   */
  async getHistoryFromDisk() {
    const getItemSize = this.getItemSize;
    const accountId = this._accountId;
    const application = this._application;
    const history = {
      deals: [],
      historyOrders: [],
      lastDealTimeByInstanceIndex: {},
      lastHistoryOrderTimeByInstanceIndex: {}
    };
    _fsExtra2.default.ensureDir('./.metaapi');
    try {
      if (await _fsExtra2.default.pathExists(`./.metaapi/${accountId}-${application}-config.bin`)) {
        let config = JSON.parse((await _fsExtra2.default.readFile(`./.metaapi/${accountId}-${application}-config.bin`, 'utf-8')).toString('utf-8'));
        history.lastDealTimeByInstanceIndex = config.lastDealTimeByInstanceIndex;
        history.lastHistoryOrderTimeByInstanceIndex = config.lastHistoryOrderTimeByInstanceIndex;
      }
    } catch (err) {
      this._logger.error(`${accountId}: Failed to read history storage config`, err);
      await _fsExtra2.default.remove(`./.metaapi/${accountId}-${application}-config.bin`);
    }
    try {
      if (await _fsExtra2.default.pathExists(`./.metaapi/${accountId}-${application}-deals.bin`)) {
        let deals = JSON.parse((await _fsExtra2.default.readFile(`./.metaapi/${accountId}-${application}-deals.bin`, 'utf-8')).toString('utf-8'));
        this._dealsSize = deals.map(deal => getItemSize(deal));
        history.deals = deals.map(deal => {
          deal.time = new Date(deal.time);
          return deal;
        });
      }
    } catch (err) {
      this._logger.error(`${accountId}: Failed to read deals history storage`, err);
      await _fsExtra2.default.remove(`./.metaapi/${accountId}-${application}-deals.bin`);
    }
    try {
      if (await _fsExtra2.default.pathExists(`./.metaapi/${accountId}-${application}-historyOrders.bin`)) {
        let historyOrders = JSON.parse((await _fsExtra2.default.readFile(`./.metaapi/${accountId}-${application}-historyOrders.bin`, 'utf-8')).toString('utf-8'));
        this._historyOrdersSize = historyOrders.map(historyOrder => getItemSize(historyOrder));
        history.historyOrders = historyOrders.map(historyOrder => {
          historyOrder.time = new Date(historyOrder.time);
          historyOrder.doneTime = new Date(historyOrder.doneTime);
          return historyOrder;
        });
      }
    } catch (err) {
      this._logger.error(`${accountId} Failed to read historyOrders history storage`, err);
      await _fsExtra2.default.remove(`./.metaapi/${accountId}-${application}-historyOrders.bin`);
    }
    return history;
  }

  /**
   * Saves unsaved history items to disk storage
   */
  async updateDiskStorage() {
    _fsExtra2.default.ensureDir('./.metaapi');
    const getItemSize = this.getItemSize;
    const accountId = this._accountId;
    const application = this._application;
    const historyStorage = this._historyStorage;
    async function replaceRecords(type, startIndex, replaceItems, sizeArray) {
      const filePath = `./.metaapi/${accountId}-${application}-${type}.bin`;
      let fileSize = (await _fsExtra2.default.stat(filePath)).size;
      if (startIndex === 0) {
        await _fsExtra2.default.writeFile(filePath, (0, _stringify2.default)(replaceItems), 'utf-8');
      } else {
        const replacedItems = sizeArray.slice(startIndex);
        // replacedItems.length - skip commas, replacedItems.reduce - skip item sizes, 1 - skip ] at the end
        const startPosition = fileSize - replacedItems.length - replacedItems.reduce((a, b) => a + b, 0) - 1;
        await _fsExtra2.default.truncate(filePath, startPosition);
        await _fsExtra2.default.appendFile(filePath, ',' + (0, _stringify2.default)(replaceItems).slice(1), { encoding: 'utf-8' });
      }
      return sizeArray.slice(0, startIndex).concat(replaceItems.map(item => getItemSize(item)));
    }

    if (!this._isUpdating) {
      this._isUpdating = true;
      try {
        await this._updateConfig();
        if (this._startNewDealIndex !== -1) {
          const filePath = `./.metaapi/${accountId}-${application}-deals.bin`;
          if (!(await _fsExtra2.default.pathExists(filePath))) {
            const deals = (0, _stringify2.default)(historyStorage.deals);
            _fsExtra2.default.writeFile(filePath, deals, 'utf-8', err => {
              if (err) {
                this._logger.error(`${accountId}: Error saving deals on disk`, err);
              }
            });
            this._dealsSize = historyStorage.deals.map(item => getItemSize(item));
          } else {
            const replaceDeals = historyStorage.deals.slice(this._startNewDealIndex);
            this._dealsSize = await replaceRecords('deals', this._startNewDealIndex, replaceDeals, this._dealsSize);
          }
          this._startNewDealIndex = -1;
        }
        if (this._startNewOrderIndex !== -1) {
          const filePath = `./.metaapi/${accountId}-${application}-historyOrders.bin`;
          if (!(await _fsExtra2.default.pathExists(filePath))) {
            const historyOrders = (0, _stringify2.default)(historyStorage.historyOrders);
            _fsExtra2.default.writeFile(filePath, historyOrders, 'utf-8', err => {
              if (err) {
                this._logger.error(`${accountId}: Error saving historyOrders on disk`, err);
              }
            });
            this._historyOrdersSize = historyStorage.historyOrders.map(item => getItemSize(item));
          } else {
            const replaceOrders = historyStorage.historyOrders.slice(this._startNewOrderIndex);
            this._historyOrdersSize = await replaceRecords('historyOrders', this._startNewOrderIndex, replaceOrders, this._historyOrdersSize);
          }
          this._startNewOrderIndex = -1;
        }
      } catch (err) {
        this._logger.error(`${accountId}: Error updating disk storage`, err);
      }
      this._isUpdating = false;
    }
  }

  /**
   * Updates stored config for account
   */
  async _updateConfig() {
    const accountId = this._accountId;
    const application = this._application;
    const historyStorage = this._historyStorage;
    const filePath = `./.metaapi/${accountId}-${application}-config.bin`;
    try {
      const config = {
        lastDealTimeByInstanceIndex: historyStorage.lastDealTimeByInstanceIndex,
        lastHistoryOrderTimeByInstanceIndex: historyStorage.lastHistoryOrderTimeByInstanceIndex
      };
      await _fsExtra2.default.writeFile(filePath, (0, _stringify2.default)(config), 'utf-8');
    } catch (err) {
      this._logger.error(`${accountId}: Error updating disk storage config`, err);
    }
  }

  /**
   * Deletes storage files from disk
   */
  async deleteStorageFromDisk() {
    await _fsExtra2.default.remove(`./.metaapi/${this._accountId}-${this._application}-config.bin`);
    await _fsExtra2.default.remove(`./.metaapi/${this._accountId}-${this._application}-deals.bin`);
    await _fsExtra2.default.remove(`./.metaapi/${this._accountId}-${this._application}-historyOrders.bin`);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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