'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _errorHandler = require('./errorHandler');

/**
 * Class for validating API options.
 */
class OptionsValidator {

  /**
   * Validates a number parameter
   * @param {Number} value value to validate
   * @param {Number} defaultValue default value for an option
   * @param {String} name option name
   * @returns {Number} validated value
   * @throws {ValidationError} if value is invalid
   */
  validateNumber(value, defaultValue, name) {
    if (value === undefined) {
      return defaultValue;
    }
    if (typeof value !== 'number') {
      throw new _errorHandler.ValidationError(`Parameter ${name} must be a number`);
    }
    if (value < 0) {
      throw new _errorHandler.ValidationError(`Parameter ${name} cannot be lower than 0`);
    }
    return value;
  }

  /**
   * Validates a number parameter to be above zero
   * @param {Number} value value to validate
   * @param {Number} defaultValue default value for an option
   * @param {String} name option name
   * @returns {Number} validated value
   * @throws {ValidationError} if value is invalid
   */
  validateNonZero(value, defaultValue, name) {
    value = this.validateNumber(value, defaultValue, name);

    if (value === 0) {
      throw new _errorHandler.ValidationError(`Parameter ${name} must be bigger than 0`);
    }
    return value;
  }

  /**
   * Validates a parameter to be boolean
   * @param {Boolean} value value to validate
   * @param {Boolean} defaultValue default value for an option
   * @param {String} name option name
   * @returns {Boolean} validated value
   * @throws {ValidationError} if value is invalid
   */
  validateBoolean(value, defaultValue, name) {
    if (value === undefined) {
      return defaultValue;
    }

    if (typeof value !== 'boolean') {
      throw new _errorHandler.ValidationError(`Parameter ${name} must be a boolean`);
    }
    return value;
  }
}
exports.default = OptionsValidator;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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