'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _timeoutError = require('../timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _optionsValidator = require('../optionsValidator');

var _optionsValidator2 = _interopRequireDefault(_optionsValidator);

var _logger = require('../../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Options for synchronization throttler
 * @typedef {Object} SynchronizationThrottlerOpts
 * @property {Number} [maxConcurrentSynchronizations] amount of maximum allowed concurrent synchronizations
 * @property {Number} [queueTimeoutInSeconds] allowed time for a synchronization in queue
 * @property {Number} [synchronizationTimeoutInSeconds] time after which a synchronization slot
 * is freed to be used by another synchronization
 */

/**
 * Synchronization throttler used to limit the amount of concurrent synchronizations to prevent application
 * from being overloaded due to excessive number of synchronisation responses being sent.
 */
class SynchronizationThrottler {

  /**
   * Constructs the synchronization throttler
   * @param {MetaApiWebsocketClient} client MetaApi websocket client
   * @param {Number} socketInstanceIndex index of socket instance that uses the throttler
   * @param {SynchronizationThrottlerOpts} opts synchronization throttler options
   */
  constructor(client, socketInstanceIndex, opts) {
    const validator = new _optionsValidator2.default();
    opts = opts || {};
    this._maxConcurrentSynchronizations = validator.validateNonZero(opts.maxConcurrentSynchronizations, 15, 'synchronizationThrottler.maxConcurrentSynchronizations');
    this._queueTimeoutInSeconds = validator.validateNonZero(opts.queueTimeoutInSeconds, 300, 'synchronizationThrottler.queueTimeoutInSeconds');
    this._synchronizationTimeoutInSeconds = validator.validateNonZero(opts.synchronizationTimeoutInSeconds, 10, 'synchronizationThrottler.synchronizationTimeoutInSeconds');
    this._client = client;
    this._socketInstanceIndex = socketInstanceIndex;
    this._synchronizationIds = {};
    this._accountsBySynchronizationIds = {};
    this._synchronizationQueue = [];
    this._removeOldSyncIdsInterval = null;
    this._processQueueInterval = null;
    this._logger = _logger2.default.getLogger('SynchronizationThrottler');
  }

  /**
   * Initializes the synchronization throttler
   */
  start() {
    if (!this._removeOldSyncIdsInterval) {
      this._removeOldSyncIdsInterval = setInterval(() => this._removeOldSyncIdsJob(), 1000);
      this._processQueueInterval = setInterval(() => this._processQueueJob(), 1000);
    }
  }

  /**
   * Deinitializes the throttler
   */
  stop() {
    clearInterval(this._removeOldSyncIdsInterval);
    this._removeOldSyncIdsInterval = null;
    clearInterval(this._processQueueInterval);
    this._processQueueInterval = null;
  }

  async _removeOldSyncIdsJob() {
    const now = Date.now();
    for (let key of (0, _keys2.default)(this._synchronizationIds)) {
      if (now - this._synchronizationIds[key] > this._synchronizationTimeoutInSeconds * 1000) {
        delete this._synchronizationIds[key];
      }
    }
    while (this._synchronizationQueue.length && Date.now() - this._synchronizationQueue[0].queueTime > this._queueTimeoutInSeconds * 1000) {
      this._removeFromQueue(this._synchronizationQueue[0].synchronizationId, 'timeout');
    }
    this._advanceQueue();
  }

  /**
   * Fills a synchronization slot with synchronization id
   * @param {String} synchronizationId synchronization id
   */
  updateSynchronizationId(synchronizationId) {
    if (this._accountsBySynchronizationIds[synchronizationId]) {
      this._synchronizationIds[synchronizationId] = Date.now();
    }
  }

  /**
   * Returns the list of currently synchronizing account ids
   */
  get synchronizingAccounts() {
    const synchronizingAccounts = [];
    (0, _keys2.default)(this._synchronizationIds).forEach(key => {
      const accountData = this._accountsBySynchronizationIds[key];
      if (accountData && !synchronizingAccounts.includes(accountData.accountId)) {
        synchronizingAccounts.push(accountData.accountId);
      }
    });
    return synchronizingAccounts;
  }

  /**
   * Returns the list of currenly active synchronization ids
   * @return {String[]} synchronization ids
   */
  get activeSynchronizationIds() {
    return (0, _keys2.default)(this._accountsBySynchronizationIds);
  }

  /**
   * Returns the amount of maximum allowed concurrent synchronizations
   * @return {number} maximum allowed concurrent synchronizations
   */
  get maxConcurrentSynchronizations() {
    const calculatedMax = Math.max(Math.ceil(this._client.subscribedAccountIds(this._socketInstanceIndex).length / 10), 1);
    return Math.min(calculatedMax, this._maxConcurrentSynchronizations);
  }

  /**
   * Returns flag whether there are free slots for synchronization requests
   * @return {Boolean} flag whether there are free slots for synchronization requests
   */
  get isSynchronizationAvailable() {
    if (this._client.socketInstances.reduce((acc, socketInstance) => acc + socketInstance.synchronizationThrottler.synchronizingAccounts.length, 0) >= this._maxConcurrentSynchronizations) {
      return false;
    }
    return this.synchronizingAccounts.length < this.maxConcurrentSynchronizations;
  }

  /**
   * Removes synchronizations from queue and from the list by parameters
   * @param {String} accountId account id
   * @param {Number} instanceIndex account instance index
   * @param {String} host account host name
   */
  removeIdByParameters(accountId, instanceIndex, host) {
    for (let key of (0, _keys2.default)(this._accountsBySynchronizationIds)) {
      if (this._accountsBySynchronizationIds[key].accountId === accountId && this._accountsBySynchronizationIds[key].instanceIndex === instanceIndex && this._accountsBySynchronizationIds[key].host === host) {
        this.removeSynchronizationId(key);
      }
    }
  }

  /**
   * Removes synchronization id from slots and removes ids for the same account from the queue
   * @param {String} synchronizationId synchronization id
   */
  removeSynchronizationId(synchronizationId) {
    if (this._accountsBySynchronizationIds[synchronizationId]) {
      const accountId = this._accountsBySynchronizationIds[synchronizationId].accountId;
      const instanceIndex = this._accountsBySynchronizationIds[synchronizationId].instanceIndex;
      const host = this._accountsBySynchronizationIds[synchronizationId].host;
      for (let key of (0, _keys2.default)(this._accountsBySynchronizationIds)) {
        if (this._accountsBySynchronizationIds[key].accountId === accountId && this._accountsBySynchronizationIds[key].instanceIndex === instanceIndex && this._accountsBySynchronizationIds[key].host === host) {
          this._removeFromQueue(key, 'cancel');
          delete this._accountsBySynchronizationIds[key];
        }
      }
    }
    if (this._synchronizationIds[synchronizationId]) {
      delete this._synchronizationIds[synchronizationId];
    }
    this._advanceQueue();
  }

  /**
   * Clears synchronization ids on disconnect
   */
  onDisconnect() {
    this._synchronizationQueue.forEach(synchronization => {
      synchronization.resolve('cancel');
    });
    this._synchronizationIds = {};
    this._accountsBySynchronizationIds = {};
    this._synchronizationQueue = [];
    this.stop();
    this.start();
  }

  _advanceQueue() {
    let index = 0;
    while (this.isSynchronizationAvailable && this._synchronizationQueue.length && index < this._synchronizationQueue.length) {
      const queueItem = this._synchronizationQueue[index];
      queueItem.resolve('synchronize');
      this.updateSynchronizationId(queueItem.synchronizationId);
      index++;
    }
  }

  _removeFromQueue(synchronizationId, result) {
    this._synchronizationQueue.forEach((syncItem, i) => {
      if (syncItem.synchronizationId === synchronizationId) {
        syncItem.resolve(result);
      }
    });
    this._synchronizationQueue = this._synchronizationQueue.filter(item => item.synchronizationId !== synchronizationId);
  }

  async _processQueueJob() {
    try {
      while (this._synchronizationQueue.length) {
        const queueItem = this._synchronizationQueue[0];
        await this._synchronizationQueue[0].promise;
        if (this._synchronizationQueue.length && this._synchronizationQueue[0].synchronizationId === queueItem.synchronizationId) {
          this._synchronizationQueue.shift();
        }
      }
    } catch (err) {
      this._logger.error('Error processing queue job', err);
    }
  }

  /**
   * Schedules to send a synchronization request for account
   * @param {String} accountId account id
   * @param {Object} request request to send
   */
  async scheduleSynchronize(accountId, request) {
    const synchronizationId = request.requestId;
    for (let key of (0, _keys2.default)(this._accountsBySynchronizationIds)) {
      if (this._accountsBySynchronizationIds[key].accountId === accountId && this._accountsBySynchronizationIds[key].instanceIndex === request.instanceIndex && this._accountsBySynchronizationIds[key].host === request.host) {
        this.removeSynchronizationId(key);
      }
    }
    this._accountsBySynchronizationIds[synchronizationId] = { accountId, instanceIndex: request.instanceIndex,
      host: request.host };
    if (!this.isSynchronizationAvailable) {
      let resolve;
      let requestResolve = new _promise2.default(res => {
        resolve = res;
      });
      this._synchronizationQueue.push({
        synchronizationId: synchronizationId,
        promise: requestResolve,
        resolve,
        queueTime: Date.now()
      });
      const result = await requestResolve;
      if (result === 'cancel') {
        return false;
      } else if (result === 'timeout') {
        throw new _timeoutError2.default(`Account ${accountId} synchronization ${synchronizationId}` + ' timed out in synchronization queue');
      }
    }
    this.updateSynchronizationId(synchronizationId);
    await this._client.rpcRequest(accountId, request);
    return true;
  }

}
exports.default = SynchronizationThrottler;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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