'use strict';

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _provisioningProfile = require('./provisioningProfile.client');

var _provisioningProfile2 = _interopRequireDefault(_provisioningProfile);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const provisioningApiUrl = 'https://mt-provisioning-api-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {ProvisioningProfileClient}
 */
describe('ProvisioningProfileClient', () => {

  let provisioningClient;
  const token = 'header.payload.sign';
  let httpClient = new _httpClient2.default();
  let sandbox;
  let requestStub;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    provisioningClient = new _provisioningProfile2.default(httpClient, token);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {ProvisioningProfileClient#getProvisioningProfiles}
   */
  it('should retrieve provisioning profiles from API', async () => {
    let expected = [{
      _id: 'id',
      name: 'name',
      version: 4,
      status: 'active'
    }];
    requestStub.resolves(expected);
    let profiles = await provisioningClient.getProvisioningProfiles(5, 'active');
    profiles.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/provisioning-profiles`,
      method: 'GET',
      qs: {
        version: 5,
        status: 'active'
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#getProvisioningProfiles}
   */
  it('should not retrieve provisioning profiles from API with account token', async () => {
    provisioningClient = new _provisioningProfile2.default(httpClient, 'token');
    try {
      await provisioningClient.getProvisioningProfiles(5, 'active');
    } catch (error) {
      error.message.should.equal('You can not invoke getProvisioningProfiles method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ProvisioningProfileClient#getProvisioningProfile}
   */
  it('should retrieve provisioning profile from API', async () => {
    let expected = {
      _id: 'id',
      name: 'name',
      version: 4,
      status: 'active'
    };
    requestStub.resolves(expected);
    let profile = await provisioningClient.getProvisioningProfile('id');
    profile.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/provisioning-profiles/id`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#getProvisioningProfile}
   */
  it('should not retrieve provisioning profile from API with account token', async () => {
    provisioningClient = new _provisioningProfile2.default(httpClient, 'token');
    try {
      await provisioningClient.getProvisioningProfile('id');
    } catch (error) {
      error.message.should.equal('You can not invoke getProvisioningProfile method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ProvisioningProfileClient#createProvisioningProfile}
   */
  it('should create provisioning profile via API', async () => {
    let expected = {
      id: 'id'
    };
    let profile = {
      name: 'name',
      version: 4
    };
    requestStub.resolves(expected);
    let id = await provisioningClient.createProvisioningProfile(profile);
    id.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/provisioning-profiles`,
      method: 'POST',
      body: profile,
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#createProvisioningProfile}
   */
  it('should not create provisioning profile via API with account token', async () => {
    provisioningClient = new _provisioningProfile2.default(httpClient, 'token');
    try {
      await provisioningClient.createProvisioningProfile({});
    } catch (error) {
      error.message.should.equal('You can not invoke createProvisioningProfile method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ProvisioningProfileClient#uploadProvisioningProfileFile}
   */
  it('should upload file to a provisioning profile via API', async () => {
    let file = Buffer.from('test', 'utf8');
    await provisioningClient.uploadProvisioningProfileFile('id', 'servers.dat', file);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/provisioning-profiles/id/servers.dat`,
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      formData: {
        file: {
          options: {
            filename: 'serverFile'
          },
          value: file
        }
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#uploadProvisioningProfileFile}
   */
  it('should not upload provisioning profile file via API with account token', async () => {
    provisioningClient = new _provisioningProfile2.default(httpClient, 'token');
    try {
      await provisioningClient.uploadProvisioningProfileFile('id', 'servers.dat', {});
    } catch (error) {
      error.message.should.equal('You can not invoke uploadProvisioningProfileFile method, because you have connected with account access' + ' token. Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ProvisioningProfileClient#deleteProvisioningProfile}
   */
  it('should delete provisioning profile via API', async () => {
    await provisioningClient.deleteProvisioningProfile('id');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/provisioning-profiles/id`,
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#deleteProvisioningProfile}
   */
  it('should not delete provisioning profile via API with account token', async () => {
    provisioningClient = new _provisioningProfile2.default(httpClient, 'token');
    try {
      await provisioningClient.deleteProvisioningProfile('id');
    } catch (error) {
      error.message.should.equal('You can not invoke deleteProvisioningProfile method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ProvisioningProfileClient#updateProvisioningProfile}
   */
  it('should update provisioning profile via API', async () => {
    await provisioningClient.updateProvisioningProfile('id', { name: 'new name' });
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/provisioning-profiles/id`,
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      json: true,
      body: {
        name: 'new name'
      }
    });
  });

  /**
   * @test {MetatraderAccountClient#updateProvisioningProfile}
   */
  it('should not update provisioning profile via API with account token', async () => {
    provisioningClient = new _provisioningProfile2.default(httpClient, 'token');
    try {
      await provisioningClient.updateProvisioningProfile('id', { name: 'new name' });
    } catch (error) {
      error.message.should.equal('You can not invoke updateProvisioningProfile method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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