'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _packetOrderer = require('./packetOrderer');

var _packetOrderer2 = _interopRequireDefault(_packetOrderer);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {PacketOrderer}
 */
describe('PacketOrderer', () => {

  let sandbox;
  let packetOrderer;
  let outOfOrderListener = {
    onOutOfOrderPacket: (accountId, instanceIndex, expectedSequenceNumber, actualSequenceNumber, packet) => {}
  };

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    packetOrderer = new _packetOrderer2.default(outOfOrderListener, 0.5);
    packetOrderer.start();
  });

  afterEach(() => {
    packetOrderer.stop();
    sandbox.restore();
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should return packets without a sequence number out immediately', () => {
    let packetWithoutSN = {
      type: 'authenticated',
      connectionId: 'accountId',
      accountId: 'accountId'
    };
    packetOrderer.restoreOrder(packetWithoutSN).should.deepEqual([packetWithoutSN]);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should restore packet order', () => {
    let firstPacket = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267178,
      sequenceNumber: 13,
      synchronizationId: 'synchronizationId'
    };
    let secondPacket = {
      type: 'prices',
      sequenceTimestamp: 1603124267180,
      sequenceNumber: 14
    };
    let thirdPacket = {
      type: 'accountInformation',
      sequenceTimestamp: 1603124267187,
      sequenceNumber: 15
    };
    let fourthPacket = {
      type: 'positions',
      sequenceTimestamp: 1603124267188,
      sequenceNumber: 16
    };
    packetOrderer.restoreOrder(secondPacket).should.deepEqual([]);
    packetOrderer.restoreOrder(firstPacket).should.deepEqual([firstPacket, secondPacket]);
    packetOrderer.restoreOrder(fourthPacket).should.deepEqual([]);
    packetOrderer.restoreOrder(thirdPacket).should.deepEqual([thirdPacket, fourthPacket]);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should filter out packets from previous synchronization attempt that includes synchronization start', () => {
    let previousStart = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267178,
      sequenceNumber: 13,
      synchronizationId: 'synchronizationId'
    };
    let oneOfPreviousPackets = {
      type: 'positions',
      sequenceTimestamp: 1603124267188,
      sequenceNumber: 15
    };
    let thisSpecifications = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267198,
      sequenceNumber: 1,
      synchronizationId: 'synchronizationId'
    };
    let thisSecondPacket = {
      type: 'prices',
      sequenceTimestamp: 1603124268198,
      sequenceNumber: 2
    };
    packetOrderer.restoreOrder(previousStart).should.deepEqual([previousStart]);
    packetOrderer.restoreOrder(oneOfPreviousPackets).should.deepEqual([]);
    packetOrderer.restoreOrder(thisSecondPacket).should.deepEqual([]);
    packetOrderer.restoreOrder(thisSpecifications).should.deepEqual([thisSpecifications, thisSecondPacket]);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should filter out packets from previous synchronization attempt that does not include the start', () => {
    let oneOfPreviousPackets = {
      type: 'positions',
      sequenceTimestamp: 1603124267188,
      sequenceNumber: 15
    };
    let thisStart = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267198,
      sequenceNumber: 16,
      synchronizationId: 'synchronizationId'
    };
    let thisSecondPacket = {
      type: 'prices',
      sequenceTimestamp: 1603124268198,
      sequenceNumber: 17
    };
    packetOrderer.restoreOrder(oneOfPreviousPackets).should.deepEqual([]);
    packetOrderer.restoreOrder(thisSecondPacket).should.deepEqual([]);
    packetOrderer.restoreOrder(thisStart).should.deepEqual([thisStart, thisSecondPacket]);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should pass trough duplicate packets', () => {
    let specificationsPacket = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267198,
      sequenceNumber: 16,
      synchronizationId: 'synchronizationId'
    };
    let secondPacket = {
      type: 'prices',
      sequenceTimestamp: 1603124268198,
      sequenceNumber: 17
    };
    packetOrderer.restoreOrder(specificationsPacket).should.deepEqual([specificationsPacket]);
    packetOrderer.restoreOrder(secondPacket).should.deepEqual([secondPacket]);
    packetOrderer.restoreOrder(secondPacket).should.deepEqual([secondPacket]);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should return in-order packets immediately', () => {
    let firstPacket = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267178,
      sequenceNumber: 13,
      synchronizationId: 'synchronizationId'
    };
    let secondPacket = {
      type: 'prices',
      sequenceTimestamp: 1603124267180,
      sequenceNumber: 14
    };
    let thirdPacket = {
      type: 'accountInformation',
      sequenceTimestamp: 1603124267187,
      sequenceNumber: 15
    };
    packetOrderer.restoreOrder(firstPacket).should.deepEqual([firstPacket]);
    packetOrderer.restoreOrder(secondPacket).should.deepEqual([secondPacket]);
    packetOrderer.restoreOrder(thirdPacket).should.deepEqual([thirdPacket]);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should call on-out-of-order listener only once per synchronzation attempt', async () => {
    sandbox.stub(outOfOrderListener, 'onOutOfOrderPacket').returns();
    let firstPacket = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267178,
      sequenceNumber: 13,
      synchronizationId: 'synchronizationId',
      accountId: 'accountId'
    };
    let thirdPacket = {
      type: 'orders',
      sequenceTimestamp: 1603124267193,
      sequenceNumber: 15,
      accountId: 'accountId'
    };
    packetOrderer.restoreOrder(firstPacket).should.deepEqual([firstPacket]);
    packetOrderer.restoreOrder(thirdPacket).should.deepEqual([]);
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.calledWith(outOfOrderListener.onOutOfOrderPacket, 'accountId', 0, 14, 15, thirdPacket);
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.calledOnce(outOfOrderListener.onOutOfOrderPacket);
  }).timeout(3000);

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should call on-out-of-order listener if the first packet in wait list is timed out', async () => {
    sandbox.stub(outOfOrderListener, 'onOutOfOrderPacket').returns();
    let timedOutPacket = {
      accountId: 'accountId',
      instanceId: 'accountId:0:ps-mpa-1',
      host: 'ps-mpa-1',
      instanceIndex: 0,
      sequenceNumber: 11,
      packet: {},
      receivedAt: new Date('2010-10-19T09:58:56.000Z')
    };
    let notTimedOutPacket = {
      accountId: 'accountId',
      instanceId: 'accountId:0:ps-mpa-1',
      host: 'ps-mpa-1',
      instanceIndex: 0,
      sequenceNumber: 15,
      packet: {},
      receivedAt: new Date('3015-10-19T09:58:56.000Z')
    };
    packetOrderer._sequenceNumberByInstance['accountId:0:ps-mpa-1'] = 1;
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'] = [timedOutPacket, notTimedOutPacket];
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.calledWith(outOfOrderListener.onOutOfOrderPacket, 'accountId', 0, 2, 11, timedOutPacket.packet);
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.calledOnce(outOfOrderListener.onOutOfOrderPacket);
  }).timeout(3000);

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should not call on-out-of-order listener if the first packet in wait list is not timed out', async () => {
    sandbox.stub(outOfOrderListener, 'onOutOfOrderPacket').returns();
    let timedOutPacket = {
      sequenceNumber: 11,
      packet: {},
      receivedAt: new Date('2010-10-19T09:58:56.000Z')
    };
    let notTimedOutPacket = {
      sequenceNumber: 15,
      packet: {},
      receivedAt: new Date('3015-10-19T09:58:56.000Z')
    };
    packetOrderer._sequenceNumberByInstance['accountId:0'] = 1;
    packetOrderer._packetsByInstance['accountId:0'] = [notTimedOutPacket, timedOutPacket];
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.notCalled(outOfOrderListener.onOutOfOrderPacket);
  }).timeout(3000);

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should not call on-out-of-order listener for packets that come before synchronization start', async () => {
    sandbox.stub(outOfOrderListener, 'onOutOfOrderPacket').returns();
    let outOfOrderPacket = {
      accountId: 'accountId',
      sequenceNumber: 11,
      packet: {},
      receivedAt: new Date('2010-10-19T09:58:56.000Z')
    };

    // There were no synchronization start packets
    packetOrderer._sequenceNumberByInstance['accountId:0'] = undefined;

    packetOrderer._packetsByInstance['accountId:0'] = [outOfOrderPacket];
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.notCalled(outOfOrderListener.onOutOfOrderPacket);
  }).timeout(3000);

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should maintain a fixed queue of wait list', () => {
    packetOrderer._waitListSizeLimit = 1;
    let secondPacket = {
      type: 'prices',
      sequenceTimestamp: 1603124267180,
      sequenceNumber: 14,
      accountId: 'accountId',
      host: 'ps-mpa-1'
    };
    let thirdPacket = {
      type: 'accountInformation',
      sequenceTimestamp: 1603124267187,
      sequenceNumber: 15,
      accountId: 'accountId',
      host: 'ps-mpa-1'
    };
    packetOrderer.restoreOrder(secondPacket);
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'].length.should.equal(1);
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'][0].packet.should.equal(secondPacket);
    packetOrderer.restoreOrder(thirdPacket);
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'].length.should.equal(1);
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'][0].packet.should.equal(thirdPacket);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should count start packets with undefined synchronziationId as out-of-order', () => {
    let startPacket = {
      type: 'synchronizationStarted',
      sequenceTimestamp: 1603124267198,
      sequenceNumber: 16,
      accountId: 'accountId',
      host: 'ps-mpa-1'
    };
    packetOrderer.restoreOrder(startPacket).should.deepEqual([]);
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'].length.should.equal(1);
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'][0].packet.should.deepEqual(startPacket);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should reset state on reconnected event', async () => {
    sandbox.stub(outOfOrderListener, 'onOutOfOrderPacket').returns();
    let timedOutPacket = {
      accountId: 'accountId',
      instanceId: 'accountId:0:ps-mpa-1',
      host: 'ps-mpa-1',
      instanceIndex: 0,
      sequenceNumber: 11,
      packet: {},
      receivedAt: new Date('2010-10-19T09:58:56.000Z')
    };
    let notTimedOutPacket = {
      accountId: 'accountId',
      instanceId: 'accountId:0:ps-mpa-1',
      host: 'ps-mpa-1',
      instanceIndex: 0,
      sequenceNumber: 15,
      packet: {},
      receivedAt: new Date('3015-10-19T09:58:56.000Z')
    };
    packetOrderer._sequenceNumberByInstance['accountId:0:ps-mpa-1'] = 1;
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'] = [timedOutPacket, notTimedOutPacket];
    packetOrderer.onReconnected(['accountId']);
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.notCalled(outOfOrderListener.onOutOfOrderPacket);
  });

  /**
   * @test {PacketOrderer#restoreOrder}
   */
  it('should reset state for an instance on stream closed event', async () => {
    sandbox.stub(outOfOrderListener, 'onOutOfOrderPacket').returns();
    let timedOutPacket = {
      accountId: 'accountId',
      instanceId: 'accountId:0:ps-mpa-1',
      host: 'ps-mpa-1',
      instanceIndex: 0,
      sequenceNumber: 11,
      packet: {},
      receivedAt: new Date('2010-10-19T09:58:56.000Z')
    };
    let notTimedOutPacket = {
      accountId: 'accountId',
      instanceId: 'accountId:0:ps-mpa-1',
      host: 'ps-mpa-1',
      instanceIndex: 0,
      sequenceNumber: 15,
      packet: {},
      receivedAt: new Date('3015-10-19T09:58:56.000Z')
    };
    packetOrderer._sequenceNumberByInstance['accountId:0:ps-mpa-1'] = 1;
    packetOrderer._packetsByInstance['accountId:0:ps-mpa-1'] = [timedOutPacket, notTimedOutPacket];
    packetOrderer.onStreamClosed('accountId:0:ps-mpa-1');
    await new _promise2.default(res => setTimeout(res, 1000));
    _sinon2.default.assert.notCalled(outOfOrderListener.onOutOfOrderPacket);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uLy4uL2xpYi9jbGllbnRzL21ldGFBcGkvcGFja2V0T3JkZXJlci5zcGVjLmVzNiJdLCJuYW1lcyI6WyJkZXNjcmliZSIsInNhbmRib3giLCJwYWNrZXRPcmRlcmVyIiwib3V0T2ZPcmRlckxpc3RlbmVyIiwib25PdXRPZk9yZGVyUGFja2V0IiwiYWNjb3VudElkIiwiaW5zdGFuY2VJbmRleCIsImV4cGVjdGVkU2VxdWVuY2VOdW1iZXIiLCJhY3R1YWxTZXF1ZW5jZU51bWJlciIsInBhY2tldCIsImJlZm9yZSIsInNpbm9uIiwiY3JlYXRlU2FuZGJveCIsImJlZm9yZUVhY2giLCJQYWNrZXRPcmRlcmVyIiwic3RhcnQiLCJhZnRlckVhY2giLCJzdG9wIiwicmVzdG9yZSIsIml0IiwicGFja2V0V2l0aG91dFNOIiwidHlwZSIsImNvbm5lY3Rpb25JZCIsInJlc3RvcmVPcmRlciIsInNob3VsZCIsImRlZXBFcXVhbCIsImZpcnN0UGFja2V0Iiwic2VxdWVuY2VUaW1lc3RhbXAiLCJzZXF1ZW5jZU51bWJlciIsInN5bmNocm9uaXphdGlvbklkIiwic2Vjb25kUGFja2V0IiwidGhpcmRQYWNrZXQiLCJmb3VydGhQYWNrZXQiLCJwcmV2aW91c1N0YXJ0Iiwib25lT2ZQcmV2aW91c1BhY2tldHMiLCJ0aGlzU3BlY2lmaWNhdGlvbnMiLCJ0aGlzU2Vjb25kUGFja2V0IiwidGhpc1N0YXJ0Iiwic3BlY2lmaWNhdGlvbnNQYWNrZXQiLCJzdHViIiwicmV0dXJucyIsInJlcyIsInNldFRpbWVvdXQiLCJhc3NlcnQiLCJjYWxsZWRXaXRoIiwiY2FsbGVkT25jZSIsInRpbWVvdXQiLCJ0aW1lZE91dFBhY2tldCIsImluc3RhbmNlSWQiLCJob3N0IiwicmVjZWl2ZWRBdCIsIkRhdGUiLCJub3RUaW1lZE91dFBhY2tldCIsIl9zZXF1ZW5jZU51bWJlckJ5SW5zdGFuY2UiLCJfcGFja2V0c0J5SW5zdGFuY2UiLCJub3RDYWxsZWQiLCJvdXRPZk9yZGVyUGFja2V0IiwidW5kZWZpbmVkIiwiX3dhaXRMaXN0U2l6ZUxpbWl0IiwibGVuZ3RoIiwiZXF1YWwiLCJzdGFydFBhY2tldCIsIm9uUmVjb25uZWN0ZWQiLCJvblN0cmVhbUNsb3NlZCJdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztBQUVBOzs7O0FBQ0E7Ozs7QUFDQTs7Ozs7O0FBRUE7OztBQUdBQSxTQUFTLGVBQVQsRUFBMEIsTUFBTTs7QUFFOUIsTUFBSUMsT0FBSjtBQUNBLE1BQUlDLGFBQUo7QUFDQSxNQUFJQyxxQkFBcUI7QUFDdkJDLHdCQUFvQixDQUFDQyxTQUFELEVBQVlDLGFBQVosRUFBMkJDLHNCQUEzQixFQUFtREMsb0JBQW5ELEVBQXlFQyxNQUF6RSxLQUFvRixDQUFFO0FBRG5GLEdBQXpCOztBQUlBQyxTQUFPLE1BQU07QUFDWFQsY0FBVVUsZ0JBQU1DLGFBQU4sRUFBVjtBQUNELEdBRkQ7O0FBSUFDLGFBQVcsTUFBTTtBQUNmWCxvQkFBZ0IsSUFBSVksdUJBQUosQ0FBa0JYLGtCQUFsQixFQUFzQyxHQUF0QyxDQUFoQjtBQUNBRCxrQkFBY2EsS0FBZDtBQUNELEdBSEQ7O0FBS0FDLFlBQVUsTUFBTTtBQUNkZCxrQkFBY2UsSUFBZDtBQUNBaEIsWUFBUWlCLE9BQVI7QUFDRCxHQUhEOztBQUtBOzs7QUFHQUMsS0FBRyxpRUFBSCxFQUFzRSxNQUFNO0FBQzFFLFFBQUlDLGtCQUFrQjtBQUNwQkMsWUFBTSxlQURjO0FBRXBCQyxvQkFBYyxXQUZNO0FBR3BCakIsaUJBQVc7QUFIUyxLQUF0QjtBQUtBSCxrQkFBY3FCLFlBQWQsQ0FBMkJILGVBQTNCLEVBQTRDSSxNQUE1QyxDQUFtREMsU0FBbkQsQ0FBNkQsQ0FBQ0wsZUFBRCxDQUE3RDtBQUNELEdBUEQ7O0FBU0E7OztBQUdBRCxLQUFHLDZCQUFILEVBQWtDLE1BQU07QUFDdEMsUUFBSU8sY0FBYztBQUNoQkwsWUFBTSx3QkFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQixFQUhBO0FBSWhCQyx5QkFBbUI7QUFKSCxLQUFsQjtBQU1BLFFBQUlDLGVBQWU7QUFDakJULFlBQU0sUUFEVztBQUVqQk0seUJBQW1CLGFBRkY7QUFHakJDLHNCQUFnQjtBQUhDLEtBQW5CO0FBS0EsUUFBSUcsY0FBYztBQUNoQlYsWUFBTSxvQkFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQjtBQUhBLEtBQWxCO0FBS0EsUUFBSUksZUFBZTtBQUNqQlgsWUFBTSxXQURXO0FBRWpCTSx5QkFBbUIsYUFGRjtBQUdqQkMsc0JBQWdCO0FBSEMsS0FBbkI7QUFLQTFCLGtCQUFjcUIsWUFBZCxDQUEyQk8sWUFBM0IsRUFBeUNOLE1BQXpDLENBQWdEQyxTQUFoRCxDQUEwRCxFQUExRDtBQUNBdkIsa0JBQWNxQixZQUFkLENBQTJCRyxXQUEzQixFQUF3Q0YsTUFBeEMsQ0FBK0NDLFNBQS9DLENBQXlELENBQUNDLFdBQUQsRUFBY0ksWUFBZCxDQUF6RDtBQUNBNUIsa0JBQWNxQixZQUFkLENBQTJCUyxZQUEzQixFQUF5Q1IsTUFBekMsQ0FBZ0RDLFNBQWhELENBQTBELEVBQTFEO0FBQ0F2QixrQkFBY3FCLFlBQWQsQ0FBMkJRLFdBQTNCLEVBQXdDUCxNQUF4QyxDQUErQ0MsU0FBL0MsQ0FBeUQsQ0FBQ00sV0FBRCxFQUFjQyxZQUFkLENBQXpEO0FBQ0QsR0ExQkQ7O0FBNEJBOzs7QUFHQWIsS0FBRyxxR0FBSCxFQUEwRyxNQUFNO0FBQzlHLFFBQUljLGdCQUFnQjtBQUNsQlosWUFBTSx3QkFEWTtBQUVsQk0seUJBQW1CLGFBRkQ7QUFHbEJDLHNCQUFnQixFQUhFO0FBSWxCQyx5QkFBbUI7QUFKRCxLQUFwQjtBQU1BLFFBQUlLLHVCQUF1QjtBQUN6QmIsWUFBTSxXQURtQjtBQUV6Qk0seUJBQW1CLGFBRk07QUFHekJDLHNCQUFnQjtBQUhTLEtBQTNCO0FBS0EsUUFBSU8scUJBQXFCO0FBQ3ZCZCxZQUFNLHdCQURpQjtBQUV2Qk0seUJBQW1CLGFBRkk7QUFHdkJDLHNCQUFnQixDQUhPO0FBSXZCQyx5QkFBbUI7QUFKSSxLQUF6QjtBQU1BLFFBQUlPLG1CQUFtQjtBQUNyQmYsWUFBTSxRQURlO0FBRXJCTSx5QkFBbUIsYUFGRTtBQUdyQkMsc0JBQWdCO0FBSEssS0FBdkI7QUFLQTFCLGtCQUFjcUIsWUFBZCxDQUEyQlUsYUFBM0IsRUFBMENULE1BQTFDLENBQWlEQyxTQUFqRCxDQUEyRCxDQUFDUSxhQUFELENBQTNEO0FBQ0EvQixrQkFBY3FCLFlBQWQsQ0FBMkJXLG9CQUEzQixFQUFpRFYsTUFBakQsQ0FBd0RDLFNBQXhELENBQWtFLEVBQWxFO0FBQ0F2QixrQkFBY3FCLFlBQWQsQ0FBMkJhLGdCQUEzQixFQUE2Q1osTUFBN0MsQ0FBb0RDLFNBQXBELENBQThELEVBQTlEO0FBQ0F2QixrQkFBY3FCLFlBQWQsQ0FBMkJZLGtCQUEzQixFQUErQ1gsTUFBL0MsQ0FBc0RDLFNBQXRELENBQWdFLENBQUNVLGtCQUFELEVBQXFCQyxnQkFBckIsQ0FBaEU7QUFDRCxHQTNCRDs7QUE2QkE7OztBQUdBakIsS0FBRyxpR0FBSCxFQUFzRyxNQUFNO0FBQzFHLFFBQUllLHVCQUF1QjtBQUN6QmIsWUFBTSxXQURtQjtBQUV6Qk0seUJBQW1CLGFBRk07QUFHekJDLHNCQUFnQjtBQUhTLEtBQTNCO0FBS0EsUUFBSVMsWUFBWTtBQUNkaEIsWUFBTSx3QkFEUTtBQUVkTSx5QkFBbUIsYUFGTDtBQUdkQyxzQkFBZ0IsRUFIRjtBQUlkQyx5QkFBbUI7QUFKTCxLQUFoQjtBQU1BLFFBQUlPLG1CQUFtQjtBQUNyQmYsWUFBTSxRQURlO0FBRXJCTSx5QkFBbUIsYUFGRTtBQUdyQkMsc0JBQWdCO0FBSEssS0FBdkI7QUFLQTFCLGtCQUFjcUIsWUFBZCxDQUEyQlcsb0JBQTNCLEVBQWlEVixNQUFqRCxDQUF3REMsU0FBeEQsQ0FBa0UsRUFBbEU7QUFDQXZCLGtCQUFjcUIsWUFBZCxDQUEyQmEsZ0JBQTNCLEVBQTZDWixNQUE3QyxDQUFvREMsU0FBcEQsQ0FBOEQsRUFBOUQ7QUFDQXZCLGtCQUFjcUIsWUFBZCxDQUEyQmMsU0FBM0IsRUFBc0NiLE1BQXRDLENBQTZDQyxTQUE3QyxDQUF1RCxDQUFDWSxTQUFELEVBQVlELGdCQUFaLENBQXZEO0FBQ0QsR0FwQkQ7O0FBc0JBOzs7QUFHQWpCLEtBQUcsc0NBQUgsRUFBMkMsTUFBTTtBQUMvQyxRQUFJbUIsdUJBQXVCO0FBQ3pCakIsWUFBTSx3QkFEbUI7QUFFekJNLHlCQUFtQixhQUZNO0FBR3pCQyxzQkFBZ0IsRUFIUztBQUl6QkMseUJBQW1CO0FBSk0sS0FBM0I7QUFNQSxRQUFJQyxlQUFlO0FBQ2pCVCxZQUFNLFFBRFc7QUFFakJNLHlCQUFtQixhQUZGO0FBR2pCQyxzQkFBZ0I7QUFIQyxLQUFuQjtBQUtBMUIsa0JBQWNxQixZQUFkLENBQTJCZSxvQkFBM0IsRUFBaURkLE1BQWpELENBQXdEQyxTQUF4RCxDQUFrRSxDQUFDYSxvQkFBRCxDQUFsRTtBQUNBcEMsa0JBQWNxQixZQUFkLENBQTJCTyxZQUEzQixFQUF5Q04sTUFBekMsQ0FBZ0RDLFNBQWhELENBQTBELENBQUNLLFlBQUQsQ0FBMUQ7QUFDQTVCLGtCQUFjcUIsWUFBZCxDQUEyQk8sWUFBM0IsRUFBeUNOLE1BQXpDLENBQWdEQyxTQUFoRCxDQUEwRCxDQUFDSyxZQUFELENBQTFEO0FBQ0QsR0FmRDs7QUFpQkE7OztBQUdBWCxLQUFHLDRDQUFILEVBQWlELE1BQU07QUFDckQsUUFBSU8sY0FBYztBQUNoQkwsWUFBTSx3QkFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQixFQUhBO0FBSWhCQyx5QkFBbUI7QUFKSCxLQUFsQjtBQU1BLFFBQUlDLGVBQWU7QUFDakJULFlBQU0sUUFEVztBQUVqQk0seUJBQW1CLGFBRkY7QUFHakJDLHNCQUFnQjtBQUhDLEtBQW5CO0FBS0EsUUFBSUcsY0FBYztBQUNoQlYsWUFBTSxvQkFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQjtBQUhBLEtBQWxCO0FBS0ExQixrQkFBY3FCLFlBQWQsQ0FBMkJHLFdBQTNCLEVBQXdDRixNQUF4QyxDQUErQ0MsU0FBL0MsQ0FBeUQsQ0FBQ0MsV0FBRCxDQUF6RDtBQUNBeEIsa0JBQWNxQixZQUFkLENBQTJCTyxZQUEzQixFQUF5Q04sTUFBekMsQ0FBZ0RDLFNBQWhELENBQTBELENBQUNLLFlBQUQsQ0FBMUQ7QUFDQTVCLGtCQUFjcUIsWUFBZCxDQUEyQlEsV0FBM0IsRUFBd0NQLE1BQXhDLENBQStDQyxTQUEvQyxDQUF5RCxDQUFDTSxXQUFELENBQXpEO0FBQ0QsR0FwQkQ7O0FBc0JBOzs7QUFHQVosS0FBRywyRUFBSCxFQUFnRixZQUFZO0FBQzFGbEIsWUFBUXNDLElBQVIsQ0FBYXBDLGtCQUFiLEVBQWlDLG9CQUFqQyxFQUF1RHFDLE9BQXZEO0FBQ0EsUUFBSWQsY0FBYztBQUNoQkwsWUFBTSx3QkFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQixFQUhBO0FBSWhCQyx5QkFBbUIsbUJBSkg7QUFLaEJ4QixpQkFBVztBQUxLLEtBQWxCO0FBT0EsUUFBSTBCLGNBQWM7QUFDaEJWLFlBQU0sUUFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQixFQUhBO0FBSWhCdkIsaUJBQVc7QUFKSyxLQUFsQjtBQU1BSCxrQkFBY3FCLFlBQWQsQ0FBMkJHLFdBQTNCLEVBQXdDRixNQUF4QyxDQUErQ0MsU0FBL0MsQ0FBeUQsQ0FBQ0MsV0FBRCxDQUF6RDtBQUNBeEIsa0JBQWNxQixZQUFkLENBQTJCUSxXQUEzQixFQUF3Q1AsTUFBeEMsQ0FBK0NDLFNBQS9DLENBQXlELEVBQXpEO0FBQ0EsVUFBTSxzQkFBWWdCLE9BQU9DLFdBQVdELEdBQVgsRUFBZ0IsSUFBaEIsQ0FBbkIsQ0FBTjtBQUNBOUIsb0JBQU1nQyxNQUFOLENBQWFDLFVBQWIsQ0FBd0J6QyxtQkFBbUJDLGtCQUEzQyxFQUNFLFdBREYsRUFDZSxDQURmLEVBQ2tCLEVBRGxCLEVBQ3NCLEVBRHRCLEVBQzBCMkIsV0FEMUI7QUFFQSxVQUFNLHNCQUFZVSxPQUFPQyxXQUFXRCxHQUFYLEVBQWdCLElBQWhCLENBQW5CLENBQU47QUFDQTlCLG9CQUFNZ0MsTUFBTixDQUFhRSxVQUFiLENBQXdCMUMsbUJBQW1CQyxrQkFBM0M7QUFDRCxHQXRCRCxFQXNCRzBDLE9BdEJILENBc0JXLElBdEJYOztBQXdCQTs7O0FBR0EzQixLQUFHLG9GQUFILEVBQXlGLFlBQVk7QUFDbkdsQixZQUFRc0MsSUFBUixDQUFhcEMsa0JBQWIsRUFBaUMsb0JBQWpDLEVBQXVEcUMsT0FBdkQ7QUFDQSxRQUFJTyxpQkFBaUI7QUFDbkIxQyxpQkFBVyxXQURRO0FBRW5CMkMsa0JBQVksc0JBRk87QUFHbkJDLFlBQU0sVUFIYTtBQUluQjNDLHFCQUFlLENBSkk7QUFLbkJzQixzQkFBZ0IsRUFMRztBQU1uQm5CLGNBQVEsRUFOVztBQU9uQnlDLGtCQUFZLElBQUlDLElBQUosQ0FBUywwQkFBVDtBQVBPLEtBQXJCO0FBU0EsUUFBSUMsb0JBQW9CO0FBQ3RCL0MsaUJBQVcsV0FEVztBQUV0QjJDLGtCQUFZLHNCQUZVO0FBR3RCQyxZQUFNLFVBSGdCO0FBSXRCM0MscUJBQWUsQ0FKTztBQUt0QnNCLHNCQUFnQixFQUxNO0FBTXRCbkIsY0FBUSxFQU5jO0FBT3RCeUMsa0JBQVksSUFBSUMsSUFBSixDQUFTLDBCQUFUO0FBUFUsS0FBeEI7QUFTQWpELGtCQUFjbUQseUJBQWQsQ0FBd0Msc0JBQXhDLElBQWtFLENBQWxFO0FBQ0FuRCxrQkFBY29ELGtCQUFkLENBQWlDLHNCQUFqQyxJQUEyRCxDQUN6RFAsY0FEeUQsRUFFekRLLGlCQUZ5RCxDQUEzRDtBQUlBLFVBQU0sc0JBQVlYLE9BQU9DLFdBQVdELEdBQVgsRUFBZ0IsSUFBaEIsQ0FBbkIsQ0FBTjtBQUNBOUIsb0JBQU1nQyxNQUFOLENBQWFDLFVBQWIsQ0FBd0J6QyxtQkFBbUJDLGtCQUEzQyxFQUNFLFdBREYsRUFDZSxDQURmLEVBQ2tCLENBRGxCLEVBQ3FCLEVBRHJCLEVBQ3lCMkMsZUFBZXRDLE1BRHhDO0FBRUEsVUFBTSxzQkFBWWdDLE9BQU9DLFdBQVdELEdBQVgsRUFBZ0IsSUFBaEIsQ0FBbkIsQ0FBTjtBQUNBOUIsb0JBQU1nQyxNQUFOLENBQWFFLFVBQWIsQ0FBd0IxQyxtQkFBbUJDLGtCQUEzQztBQUNELEdBOUJELEVBOEJHMEMsT0E5QkgsQ0E4QlcsSUE5Qlg7O0FBZ0NBOzs7QUFHQTNCLEtBQUcsNEZBQUgsRUFBaUcsWUFBWTtBQUMzR2xCLFlBQVFzQyxJQUFSLENBQWFwQyxrQkFBYixFQUFpQyxvQkFBakMsRUFBdURxQyxPQUF2RDtBQUNBLFFBQUlPLGlCQUFpQjtBQUNuQm5CLHNCQUFnQixFQURHO0FBRW5CbkIsY0FBUSxFQUZXO0FBR25CeUMsa0JBQVksSUFBSUMsSUFBSixDQUFTLDBCQUFUO0FBSE8sS0FBckI7QUFLQSxRQUFJQyxvQkFBb0I7QUFDdEJ4QixzQkFBZ0IsRUFETTtBQUV0Qm5CLGNBQVEsRUFGYztBQUd0QnlDLGtCQUFZLElBQUlDLElBQUosQ0FBUywwQkFBVDtBQUhVLEtBQXhCO0FBS0FqRCxrQkFBY21ELHlCQUFkLENBQXdDLGFBQXhDLElBQXlELENBQXpEO0FBQ0FuRCxrQkFBY29ELGtCQUFkLENBQWlDLGFBQWpDLElBQWtELENBQ2hERixpQkFEZ0QsRUFFaERMLGNBRmdELENBQWxEO0FBSUEsVUFBTSxzQkFBWU4sT0FBT0MsV0FBV0QsR0FBWCxFQUFnQixJQUFoQixDQUFuQixDQUFOO0FBQ0E5QixvQkFBTWdDLE1BQU4sQ0FBYVksU0FBYixDQUF1QnBELG1CQUFtQkMsa0JBQTFDO0FBQ0QsR0FuQkQsRUFtQkcwQyxPQW5CSCxDQW1CVyxJQW5CWDs7QUFxQkE7OztBQUdBM0IsS0FBRyw2RkFBSCxFQUFrRyxZQUFZO0FBQzVHbEIsWUFBUXNDLElBQVIsQ0FBYXBDLGtCQUFiLEVBQWlDLG9CQUFqQyxFQUF1RHFDLE9BQXZEO0FBQ0EsUUFBSWdCLG1CQUFtQjtBQUNyQm5ELGlCQUFXLFdBRFU7QUFFckJ1QixzQkFBZ0IsRUFGSztBQUdyQm5CLGNBQVEsRUFIYTtBQUlyQnlDLGtCQUFZLElBQUlDLElBQUosQ0FBUywwQkFBVDtBQUpTLEtBQXZCOztBQU9BO0FBQ0FqRCxrQkFBY21ELHlCQUFkLENBQXdDLGFBQXhDLElBQXlESSxTQUF6RDs7QUFFQXZELGtCQUFjb0Qsa0JBQWQsQ0FBaUMsYUFBakMsSUFBa0QsQ0FBQ0UsZ0JBQUQsQ0FBbEQ7QUFDQSxVQUFNLHNCQUFZZixPQUFPQyxXQUFXRCxHQUFYLEVBQWdCLElBQWhCLENBQW5CLENBQU47QUFDQTlCLG9CQUFNZ0MsTUFBTixDQUFhWSxTQUFiLENBQXVCcEQsbUJBQW1CQyxrQkFBMUM7QUFDRCxHQWZELEVBZUcwQyxPQWZILENBZVcsSUFmWDs7QUFpQkE7OztBQUdBM0IsS0FBRyw0Q0FBSCxFQUFpRCxNQUFNO0FBQ3JEakIsa0JBQWN3RCxrQkFBZCxHQUFtQyxDQUFuQztBQUNBLFFBQUk1QixlQUFlO0FBQ2pCVCxZQUFNLFFBRFc7QUFFakJNLHlCQUFtQixhQUZGO0FBR2pCQyxzQkFBZ0IsRUFIQztBQUlqQnZCLGlCQUFXLFdBSk07QUFLakI0QyxZQUFNO0FBTFcsS0FBbkI7QUFPQSxRQUFJbEIsY0FBYztBQUNoQlYsWUFBTSxvQkFEVTtBQUVoQk0seUJBQW1CLGFBRkg7QUFHaEJDLHNCQUFnQixFQUhBO0FBSWhCdkIsaUJBQVcsV0FKSztBQUtoQjRDLFlBQU07QUFMVSxLQUFsQjtBQU9BL0Msa0JBQWNxQixZQUFkLENBQTJCTyxZQUEzQjtBQUNBNUIsa0JBQWNvRCxrQkFBZCxDQUFpQyxzQkFBakMsRUFBeURLLE1BQXpELENBQWdFbkMsTUFBaEUsQ0FBdUVvQyxLQUF2RSxDQUE2RSxDQUE3RTtBQUNBMUQsa0JBQWNvRCxrQkFBZCxDQUFpQyxzQkFBakMsRUFBeUQsQ0FBekQsRUFBNEQ3QyxNQUE1RCxDQUFtRWUsTUFBbkUsQ0FBMEVvQyxLQUExRSxDQUFnRjlCLFlBQWhGO0FBQ0E1QixrQkFBY3FCLFlBQWQsQ0FBMkJRLFdBQTNCO0FBQ0E3QixrQkFBY29ELGtCQUFkLENBQWlDLHNCQUFqQyxFQUF5REssTUFBekQsQ0FBZ0VuQyxNQUFoRSxDQUF1RW9DLEtBQXZFLENBQTZFLENBQTdFO0FBQ0ExRCxrQkFBY29ELGtCQUFkLENBQWlDLHNCQUFqQyxFQUF5RCxDQUF6RCxFQUE0RDdDLE1BQTVELENBQW1FZSxNQUFuRSxDQUEwRW9DLEtBQTFFLENBQWdGN0IsV0FBaEY7QUFDRCxHQXRCRDs7QUF3QkE7OztBQUdBWixLQUFHLDZFQUFILEVBQWtGLE1BQU07QUFDdEYsUUFBSTBDLGNBQWM7QUFDaEJ4QyxZQUFNLHdCQURVO0FBRWhCTSx5QkFBbUIsYUFGSDtBQUdoQkMsc0JBQWdCLEVBSEE7QUFJaEJ2QixpQkFBVyxXQUpLO0FBS2hCNEMsWUFBTTtBQUxVLEtBQWxCO0FBT0EvQyxrQkFBY3FCLFlBQWQsQ0FBMkJzQyxXQUEzQixFQUF3Q3JDLE1BQXhDLENBQStDQyxTQUEvQyxDQUF5RCxFQUF6RDtBQUNBdkIsa0JBQWNvRCxrQkFBZCxDQUFpQyxzQkFBakMsRUFBeURLLE1BQXpELENBQWdFbkMsTUFBaEUsQ0FBdUVvQyxLQUF2RSxDQUE2RSxDQUE3RTtBQUNBMUQsa0JBQWNvRCxrQkFBZCxDQUFpQyxzQkFBakMsRUFBeUQsQ0FBekQsRUFBNEQ3QyxNQUE1RCxDQUFtRWUsTUFBbkUsQ0FBMEVDLFNBQTFFLENBQW9Gb0MsV0FBcEY7QUFDRCxHQVhEOztBQWFBOzs7QUFHQTFDLEtBQUcseUNBQUgsRUFBOEMsWUFBWTtBQUN4RGxCLFlBQVFzQyxJQUFSLENBQWFwQyxrQkFBYixFQUFpQyxvQkFBakMsRUFBdURxQyxPQUF2RDtBQUNBLFFBQUlPLGlCQUFpQjtBQUNuQjFDLGlCQUFXLFdBRFE7QUFFbkIyQyxrQkFBWSxzQkFGTztBQUduQkMsWUFBTSxVQUhhO0FBSW5CM0MscUJBQWUsQ0FKSTtBQUtuQnNCLHNCQUFnQixFQUxHO0FBTW5CbkIsY0FBUSxFQU5XO0FBT25CeUMsa0JBQVksSUFBSUMsSUFBSixDQUFTLDBCQUFUO0FBUE8sS0FBckI7QUFTQSxRQUFJQyxvQkFBb0I7QUFDdEIvQyxpQkFBVyxXQURXO0FBRXRCMkMsa0JBQVksc0JBRlU7QUFHdEJDLFlBQU0sVUFIZ0I7QUFJdEIzQyxxQkFBZSxDQUpPO0FBS3RCc0Isc0JBQWdCLEVBTE07QUFNdEJuQixjQUFRLEVBTmM7QUFPdEJ5QyxrQkFBWSxJQUFJQyxJQUFKLENBQVMsMEJBQVQ7QUFQVSxLQUF4QjtBQVNBakQsa0JBQWNtRCx5QkFBZCxDQUF3QyxzQkFBeEMsSUFBa0UsQ0FBbEU7QUFDQW5ELGtCQUFjb0Qsa0JBQWQsQ0FBaUMsc0JBQWpDLElBQTJELENBQ3pEUCxjQUR5RCxFQUV6REssaUJBRnlELENBQTNEO0FBSUFsRCxrQkFBYzRELGFBQWQsQ0FBNEIsQ0FBQyxXQUFELENBQTVCO0FBQ0EsVUFBTSxzQkFBWXJCLE9BQU9DLFdBQVdELEdBQVgsRUFBZ0IsSUFBaEIsQ0FBbkIsQ0FBTjtBQUNBOUIsb0JBQU1nQyxNQUFOLENBQWFZLFNBQWIsQ0FBdUJwRCxtQkFBbUJDLGtCQUExQztBQUNELEdBNUJEOztBQThCQTs7O0FBR0FlLEtBQUcsMkRBQUgsRUFBZ0UsWUFBWTtBQUMxRWxCLFlBQVFzQyxJQUFSLENBQWFwQyxrQkFBYixFQUFpQyxvQkFBakMsRUFBdURxQyxPQUF2RDtBQUNBLFFBQUlPLGlCQUFpQjtBQUNuQjFDLGlCQUFXLFdBRFE7QUFFbkIyQyxrQkFBWSxzQkFGTztBQUduQkMsWUFBTSxVQUhhO0FBSW5CM0MscUJBQWUsQ0FKSTtBQUtuQnNCLHNCQUFnQixFQUxHO0FBTW5CbkIsY0FBUSxFQU5XO0FBT25CeUMsa0JBQVksSUFBSUMsSUFBSixDQUFTLDBCQUFUO0FBUE8sS0FBckI7QUFTQSxRQUFJQyxvQkFBb0I7QUFDdEIvQyxpQkFBVyxXQURXO0FBRXRCMkMsa0JBQVksc0JBRlU7QUFHdEJDLFlBQU0sVUFIZ0I7QUFJdEIzQyxxQkFBZSxDQUpPO0FBS3RCc0Isc0JBQWdCLEVBTE07QUFNdEJuQixjQUFRLEVBTmM7QUFPdEJ5QyxrQkFBWSxJQUFJQyxJQUFKLENBQVMsMEJBQVQ7QUFQVSxLQUF4QjtBQVNBakQsa0JBQWNtRCx5QkFBZCxDQUF3QyxzQkFBeEMsSUFBa0UsQ0FBbEU7QUFDQW5ELGtCQUFjb0Qsa0JBQWQsQ0FBaUMsc0JBQWpDLElBQTJELENBQ3pEUCxjQUR5RCxFQUV6REssaUJBRnlELENBQTNEO0FBSUFsRCxrQkFBYzZELGNBQWQsQ0FBNkIsc0JBQTdCO0FBQ0EsVUFBTSxzQkFBWXRCLE9BQU9DLFdBQVdELEdBQVgsRUFBZ0IsSUFBaEIsQ0FBbkIsQ0FBTjtBQUNBOUIsb0JBQU1nQyxNQUFOLENBQWFZLFNBQWIsQ0FBdUJwRCxtQkFBbUJDLGtCQUExQztBQUNELEdBNUJEO0FBNkJELENBN1hEIiwiZmlsZSI6InBhY2tldE9yZGVyZXIuc3BlYy5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxuaW1wb3J0IHNpbm9uIGZyb20gJ3Npbm9uJztcbmltcG9ydCBzaG91bGQgZnJvbSAnc2hvdWxkJztcbmltcG9ydCBQYWNrZXRPcmRlcmVyIGZyb20gJy4vcGFja2V0T3JkZXJlcic7XG5cbi8qKlxuICogQHRlc3Qge1BhY2tldE9yZGVyZXJ9XG4gKi9cbmRlc2NyaWJlKCdQYWNrZXRPcmRlcmVyJywgKCkgPT4ge1xuXG4gIGxldCBzYW5kYm94O1xuICBsZXQgcGFja2V0T3JkZXJlcjtcbiAgbGV0IG91dE9mT3JkZXJMaXN0ZW5lciA9IHtcbiAgICBvbk91dE9mT3JkZXJQYWNrZXQ6IChhY2NvdW50SWQsIGluc3RhbmNlSW5kZXgsIGV4cGVjdGVkU2VxdWVuY2VOdW1iZXIsIGFjdHVhbFNlcXVlbmNlTnVtYmVyLCBwYWNrZXQpID0+IHt9XG4gIH07XG5cbiAgYmVmb3JlKCgpID0+IHtcbiAgICBzYW5kYm94ID0gc2lub24uY3JlYXRlU2FuZGJveCgpO1xuICB9KTtcblxuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICBwYWNrZXRPcmRlcmVyID0gbmV3IFBhY2tldE9yZGVyZXIob3V0T2ZPcmRlckxpc3RlbmVyLCAwLjUpO1xuICAgIHBhY2tldE9yZGVyZXIuc3RhcnQoKTtcbiAgfSk7XG5cbiAgYWZ0ZXJFYWNoKCgpID0+IHtcbiAgICBwYWNrZXRPcmRlcmVyLnN0b3AoKTtcbiAgICBzYW5kYm94LnJlc3RvcmUoKTtcbiAgfSk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtQYWNrZXRPcmRlcmVyI3Jlc3RvcmVPcmRlcn1cbiAgICovXG4gIGl0KCdzaG91bGQgcmV0dXJuIHBhY2tldHMgd2l0aG91dCBhIHNlcXVlbmNlIG51bWJlciBvdXQgaW1tZWRpYXRlbHknLCAoKSA9PiB7XG4gICAgbGV0IHBhY2tldFdpdGhvdXRTTiA9IHtcbiAgICAgIHR5cGU6ICdhdXRoZW50aWNhdGVkJyxcbiAgICAgIGNvbm5lY3Rpb25JZDogJ2FjY291bnRJZCcsXG4gICAgICBhY2NvdW50SWQ6ICdhY2NvdW50SWQnXG4gICAgfTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcihwYWNrZXRXaXRob3V0U04pLnNob3VsZC5kZWVwRXF1YWwoW3BhY2tldFdpdGhvdXRTTl0pO1xuICB9KTtcblxuICAvKipcbiAgICogQHRlc3Qge1BhY2tldE9yZGVyZXIjcmVzdG9yZU9yZGVyfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCByZXN0b3JlIHBhY2tldCBvcmRlcicsICgpID0+IHtcbiAgICBsZXQgZmlyc3RQYWNrZXQgPSB7XG4gICAgICB0eXBlOiAnc3luY2hyb25pemF0aW9uU3RhcnRlZCcsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE3OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxMyxcbiAgICAgIHN5bmNocm9uaXphdGlvbklkOiAnc3luY2hyb25pemF0aW9uSWQnXG4gICAgfTtcbiAgICBsZXQgc2Vjb25kUGFja2V0ID0ge1xuICAgICAgdHlwZTogJ3ByaWNlcycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE4MCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNCxcbiAgICB9O1xuICAgIGxldCB0aGlyZFBhY2tldCA9IHtcbiAgICAgIHR5cGU6ICdhY2NvdW50SW5mb3JtYXRpb24nLFxuICAgICAgc2VxdWVuY2VUaW1lc3RhbXA6IDE2MDMxMjQyNjcxODcsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTUsXG4gICAgfTtcbiAgICBsZXQgZm91cnRoUGFja2V0ID0ge1xuICAgICAgdHlwZTogJ3Bvc2l0aW9ucycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE4OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNixcbiAgICB9O1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHNlY29uZFBhY2tldCkuc2hvdWxkLmRlZXBFcXVhbChbXSk7XG4gICAgcGFja2V0T3JkZXJlci5yZXN0b3JlT3JkZXIoZmlyc3RQYWNrZXQpLnNob3VsZC5kZWVwRXF1YWwoW2ZpcnN0UGFja2V0LCBzZWNvbmRQYWNrZXRdKTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcihmb3VydGhQYWNrZXQpLnNob3VsZC5kZWVwRXF1YWwoW10pO1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHRoaXJkUGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFt0aGlyZFBhY2tldCwgZm91cnRoUGFja2V0XSk7XG4gIH0pO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7UGFja2V0T3JkZXJlciNyZXN0b3JlT3JkZXJ9XG4gICAqL1xuICBpdCgnc2hvdWxkIGZpbHRlciBvdXQgcGFja2V0cyBmcm9tIHByZXZpb3VzIHN5bmNocm9uaXphdGlvbiBhdHRlbXB0IHRoYXQgaW5jbHVkZXMgc3luY2hyb25pemF0aW9uIHN0YXJ0JywgKCkgPT4ge1xuICAgIGxldCBwcmV2aW91c1N0YXJ0ID0ge1xuICAgICAgdHlwZTogJ3N5bmNocm9uaXphdGlvblN0YXJ0ZWQnLFxuICAgICAgc2VxdWVuY2VUaW1lc3RhbXA6IDE2MDMxMjQyNjcxNzgsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTMsXG4gICAgICBzeW5jaHJvbml6YXRpb25JZDogJ3N5bmNocm9uaXphdGlvbklkJ1xuICAgIH07XG4gICAgbGV0IG9uZU9mUHJldmlvdXNQYWNrZXRzID0ge1xuICAgICAgdHlwZTogJ3Bvc2l0aW9ucycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE4OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNSxcbiAgICB9O1xuICAgIGxldCB0aGlzU3BlY2lmaWNhdGlvbnMgPSB7XG4gICAgICB0eXBlOiAnc3luY2hyb25pemF0aW9uU3RhcnRlZCcsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE5OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxLFxuICAgICAgc3luY2hyb25pemF0aW9uSWQ6ICdzeW5jaHJvbml6YXRpb25JZCdcbiAgICB9O1xuICAgIGxldCB0aGlzU2Vjb25kUGFja2V0ID0ge1xuICAgICAgdHlwZTogJ3ByaWNlcycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2ODE5OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAyLFxuICAgIH07XG4gICAgcGFja2V0T3JkZXJlci5yZXN0b3JlT3JkZXIocHJldmlvdXNTdGFydCkuc2hvdWxkLmRlZXBFcXVhbChbcHJldmlvdXNTdGFydF0pO1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKG9uZU9mUHJldmlvdXNQYWNrZXRzKS5zaG91bGQuZGVlcEVxdWFsKFtdKTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcih0aGlzU2Vjb25kUGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFtdKTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcih0aGlzU3BlY2lmaWNhdGlvbnMpLnNob3VsZC5kZWVwRXF1YWwoW3RoaXNTcGVjaWZpY2F0aW9ucywgdGhpc1NlY29uZFBhY2tldF0pO1xuICB9KTtcblxuICAvKipcbiAgICogQHRlc3Qge1BhY2tldE9yZGVyZXIjcmVzdG9yZU9yZGVyfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCBmaWx0ZXIgb3V0IHBhY2tldHMgZnJvbSBwcmV2aW91cyBzeW5jaHJvbml6YXRpb24gYXR0ZW1wdCB0aGF0IGRvZXMgbm90IGluY2x1ZGUgdGhlIHN0YXJ0JywgKCkgPT4ge1xuICAgIGxldCBvbmVPZlByZXZpb3VzUGFja2V0cyA9IHtcbiAgICAgIHR5cGU6ICdwb3NpdGlvbnMnLFxuICAgICAgc2VxdWVuY2VUaW1lc3RhbXA6IDE2MDMxMjQyNjcxODgsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTUsXG4gICAgfTtcbiAgICBsZXQgdGhpc1N0YXJ0ID0ge1xuICAgICAgdHlwZTogJ3N5bmNocm9uaXphdGlvblN0YXJ0ZWQnLFxuICAgICAgc2VxdWVuY2VUaW1lc3RhbXA6IDE2MDMxMjQyNjcxOTgsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTYsXG4gICAgICBzeW5jaHJvbml6YXRpb25JZDogJ3N5bmNocm9uaXphdGlvbklkJ1xuICAgIH07XG4gICAgbGV0IHRoaXNTZWNvbmRQYWNrZXQgPSB7XG4gICAgICB0eXBlOiAncHJpY2VzJyxcbiAgICAgIHNlcXVlbmNlVGltZXN0YW1wOiAxNjAzMTI0MjY4MTk4LFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDE3LFxuICAgIH07XG4gICAgcGFja2V0T3JkZXJlci5yZXN0b3JlT3JkZXIob25lT2ZQcmV2aW91c1BhY2tldHMpLnNob3VsZC5kZWVwRXF1YWwoW10pO1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHRoaXNTZWNvbmRQYWNrZXQpLnNob3VsZC5kZWVwRXF1YWwoW10pO1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHRoaXNTdGFydCkuc2hvdWxkLmRlZXBFcXVhbChbdGhpc1N0YXJ0LCB0aGlzU2Vjb25kUGFja2V0XSk7XG4gIH0pO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7UGFja2V0T3JkZXJlciNyZXN0b3JlT3JkZXJ9XG4gICAqL1xuICBpdCgnc2hvdWxkIHBhc3MgdHJvdWdoIGR1cGxpY2F0ZSBwYWNrZXRzJywgKCkgPT4ge1xuICAgIGxldCBzcGVjaWZpY2F0aW9uc1BhY2tldCA9IHtcbiAgICAgIHR5cGU6ICdzeW5jaHJvbml6YXRpb25TdGFydGVkJyxcbiAgICAgIHNlcXVlbmNlVGltZXN0YW1wOiAxNjAzMTI0MjY3MTk4LFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDE2LFxuICAgICAgc3luY2hyb25pemF0aW9uSWQ6ICdzeW5jaHJvbml6YXRpb25JZCdcbiAgICB9O1xuICAgIGxldCBzZWNvbmRQYWNrZXQgPSB7XG4gICAgICB0eXBlOiAncHJpY2VzJyxcbiAgICAgIHNlcXVlbmNlVGltZXN0YW1wOiAxNjAzMTI0MjY4MTk4LFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDE3LFxuICAgIH07XG4gICAgcGFja2V0T3JkZXJlci5yZXN0b3JlT3JkZXIoc3BlY2lmaWNhdGlvbnNQYWNrZXQpLnNob3VsZC5kZWVwRXF1YWwoW3NwZWNpZmljYXRpb25zUGFja2V0XSk7XG4gICAgcGFja2V0T3JkZXJlci5yZXN0b3JlT3JkZXIoc2Vjb25kUGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFtzZWNvbmRQYWNrZXRdKTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcihzZWNvbmRQYWNrZXQpLnNob3VsZC5kZWVwRXF1YWwoW3NlY29uZFBhY2tldF0pO1xuICB9KTtcblxuICAvKipcbiAgICogQHRlc3Qge1BhY2tldE9yZGVyZXIjcmVzdG9yZU9yZGVyfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCByZXR1cm4gaW4tb3JkZXIgcGFja2V0cyBpbW1lZGlhdGVseScsICgpID0+IHtcbiAgICBsZXQgZmlyc3RQYWNrZXQgPSB7XG4gICAgICB0eXBlOiAnc3luY2hyb25pemF0aW9uU3RhcnRlZCcsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE3OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxMyxcbiAgICAgIHN5bmNocm9uaXphdGlvbklkOiAnc3luY2hyb25pemF0aW9uSWQnXG4gICAgfTtcbiAgICBsZXQgc2Vjb25kUGFja2V0ID0ge1xuICAgICAgdHlwZTogJ3ByaWNlcycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE4MCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNCxcbiAgICB9O1xuICAgIGxldCB0aGlyZFBhY2tldCA9IHtcbiAgICAgIHR5cGU6ICdhY2NvdW50SW5mb3JtYXRpb24nLFxuICAgICAgc2VxdWVuY2VUaW1lc3RhbXA6IDE2MDMxMjQyNjcxODcsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTUsXG4gICAgfTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcihmaXJzdFBhY2tldCkuc2hvdWxkLmRlZXBFcXVhbChbZmlyc3RQYWNrZXRdKTtcbiAgICBwYWNrZXRPcmRlcmVyLnJlc3RvcmVPcmRlcihzZWNvbmRQYWNrZXQpLnNob3VsZC5kZWVwRXF1YWwoW3NlY29uZFBhY2tldF0pO1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHRoaXJkUGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFt0aGlyZFBhY2tldF0pO1xuICB9KTtcblxuICAvKipcbiAgICogQHRlc3Qge1BhY2tldE9yZGVyZXIjcmVzdG9yZU9yZGVyfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCBjYWxsIG9uLW91dC1vZi1vcmRlciBsaXN0ZW5lciBvbmx5IG9uY2UgcGVyIHN5bmNocm9uemF0aW9uIGF0dGVtcHQnLCBhc3luYyAoKSA9PiB7XG4gICAgc2FuZGJveC5zdHViKG91dE9mT3JkZXJMaXN0ZW5lciwgJ29uT3V0T2ZPcmRlclBhY2tldCcpLnJldHVybnMoKTtcbiAgICBsZXQgZmlyc3RQYWNrZXQgPSB7XG4gICAgICB0eXBlOiAnc3luY2hyb25pemF0aW9uU3RhcnRlZCcsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE3OCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxMyxcbiAgICAgIHN5bmNocm9uaXphdGlvbklkOiAnc3luY2hyb25pemF0aW9uSWQnLFxuICAgICAgYWNjb3VudElkOiAnYWNjb3VudElkJ1xuICAgIH07XG4gICAgbGV0IHRoaXJkUGFja2V0ID0ge1xuICAgICAgdHlwZTogJ29yZGVycycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE5MyxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNSxcbiAgICAgIGFjY291bnRJZDogJ2FjY291bnRJZCdcbiAgICB9O1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKGZpcnN0UGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFtmaXJzdFBhY2tldF0pO1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHRoaXJkUGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFtdKTtcbiAgICBhd2FpdCBuZXcgUHJvbWlzZShyZXMgPT4gc2V0VGltZW91dChyZXMsIDEwMDApKTtcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkV2l0aChvdXRPZk9yZGVyTGlzdGVuZXIub25PdXRPZk9yZGVyUGFja2V0LFxuICAgICAgJ2FjY291bnRJZCcsIDAsIDE0LCAxNSwgdGhpcmRQYWNrZXQpO1xuICAgIGF3YWl0IG5ldyBQcm9taXNlKHJlcyA9PiBzZXRUaW1lb3V0KHJlcywgMTAwMCkpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRPbmNlKG91dE9mT3JkZXJMaXN0ZW5lci5vbk91dE9mT3JkZXJQYWNrZXQpO1xuICB9KS50aW1lb3V0KDMwMDApO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7UGFja2V0T3JkZXJlciNyZXN0b3JlT3JkZXJ9XG4gICAqL1xuICBpdCgnc2hvdWxkIGNhbGwgb24tb3V0LW9mLW9yZGVyIGxpc3RlbmVyIGlmIHRoZSBmaXJzdCBwYWNrZXQgaW4gd2FpdCBsaXN0IGlzIHRpbWVkIG91dCcsIGFzeW5jICgpID0+IHtcbiAgICBzYW5kYm94LnN0dWIob3V0T2ZPcmRlckxpc3RlbmVyLCAnb25PdXRPZk9yZGVyUGFja2V0JykucmV0dXJucygpO1xuICAgIGxldCB0aW1lZE91dFBhY2tldCA9IHtcbiAgICAgIGFjY291bnRJZDogJ2FjY291bnRJZCcsXG4gICAgICBpbnN0YW5jZUlkOiAnYWNjb3VudElkOjA6cHMtbXBhLTEnLFxuICAgICAgaG9zdDogJ3BzLW1wYS0xJyxcbiAgICAgIGluc3RhbmNlSW5kZXg6IDAsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTEsXG4gICAgICBwYWNrZXQ6IHt9LFxuICAgICAgcmVjZWl2ZWRBdDogbmV3IERhdGUoJzIwMTAtMTAtMTlUMDk6NTg6NTYuMDAwWicpXG4gICAgfTtcbiAgICBsZXQgbm90VGltZWRPdXRQYWNrZXQgPSB7XG4gICAgICBhY2NvdW50SWQ6ICdhY2NvdW50SWQnLFxuICAgICAgaW5zdGFuY2VJZDogJ2FjY291bnRJZDowOnBzLW1wYS0xJyxcbiAgICAgIGhvc3Q6ICdwcy1tcGEtMScsXG4gICAgICBpbnN0YW5jZUluZGV4OiAwLFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDE1LFxuICAgICAgcGFja2V0OiB7fSxcbiAgICAgIHJlY2VpdmVkQXQ6IG5ldyBEYXRlKCczMDE1LTEwLTE5VDA5OjU4OjU2LjAwMFonKVxuICAgIH07XG4gICAgcGFja2V0T3JkZXJlci5fc2VxdWVuY2VOdW1iZXJCeUluc3RhbmNlWydhY2NvdW50SWQ6MDpwcy1tcGEtMSddID0gMTtcbiAgICBwYWNrZXRPcmRlcmVyLl9wYWNrZXRzQnlJbnN0YW5jZVsnYWNjb3VudElkOjA6cHMtbXBhLTEnXSA9IFtcbiAgICAgIHRpbWVkT3V0UGFja2V0LFxuICAgICAgbm90VGltZWRPdXRQYWNrZXRcbiAgICBdO1xuICAgIGF3YWl0IG5ldyBQcm9taXNlKHJlcyA9PiBzZXRUaW1lb3V0KHJlcywgMTAwMCkpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRXaXRoKG91dE9mT3JkZXJMaXN0ZW5lci5vbk91dE9mT3JkZXJQYWNrZXQsXG4gICAgICAnYWNjb3VudElkJywgMCwgMiwgMTEsIHRpbWVkT3V0UGFja2V0LnBhY2tldCk7XG4gICAgYXdhaXQgbmV3IFByb21pc2UocmVzID0+IHNldFRpbWVvdXQocmVzLCAxMDAwKSk7XG4gICAgc2lub24uYXNzZXJ0LmNhbGxlZE9uY2Uob3V0T2ZPcmRlckxpc3RlbmVyLm9uT3V0T2ZPcmRlclBhY2tldCk7XG4gIH0pLnRpbWVvdXQoMzAwMCk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtQYWNrZXRPcmRlcmVyI3Jlc3RvcmVPcmRlcn1cbiAgICovXG4gIGl0KCdzaG91bGQgbm90IGNhbGwgb24tb3V0LW9mLW9yZGVyIGxpc3RlbmVyIGlmIHRoZSBmaXJzdCBwYWNrZXQgaW4gd2FpdCBsaXN0IGlzIG5vdCB0aW1lZCBvdXQnLCBhc3luYyAoKSA9PiB7XG4gICAgc2FuZGJveC5zdHViKG91dE9mT3JkZXJMaXN0ZW5lciwgJ29uT3V0T2ZPcmRlclBhY2tldCcpLnJldHVybnMoKTtcbiAgICBsZXQgdGltZWRPdXRQYWNrZXQgPSB7XG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTEsXG4gICAgICBwYWNrZXQ6IHt9LFxuICAgICAgcmVjZWl2ZWRBdDogbmV3IERhdGUoJzIwMTAtMTAtMTlUMDk6NTg6NTYuMDAwWicpXG4gICAgfTtcbiAgICBsZXQgbm90VGltZWRPdXRQYWNrZXQgPSB7XG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTUsXG4gICAgICBwYWNrZXQ6IHt9LFxuICAgICAgcmVjZWl2ZWRBdDogbmV3IERhdGUoJzMwMTUtMTAtMTlUMDk6NTg6NTYuMDAwWicpXG4gICAgfTtcbiAgICBwYWNrZXRPcmRlcmVyLl9zZXF1ZW5jZU51bWJlckJ5SW5zdGFuY2VbJ2FjY291bnRJZDowJ10gPSAxO1xuICAgIHBhY2tldE9yZGVyZXIuX3BhY2tldHNCeUluc3RhbmNlWydhY2NvdW50SWQ6MCddID0gW1xuICAgICAgbm90VGltZWRPdXRQYWNrZXQsXG4gICAgICB0aW1lZE91dFBhY2tldFxuICAgIF07XG4gICAgYXdhaXQgbmV3IFByb21pc2UocmVzID0+IHNldFRpbWVvdXQocmVzLCAxMDAwKSk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChvdXRPZk9yZGVyTGlzdGVuZXIub25PdXRPZk9yZGVyUGFja2V0KTtcbiAgfSkudGltZW91dCgzMDAwKTtcblxuICAvKipcbiAgICogQHRlc3Qge1BhY2tldE9yZGVyZXIjcmVzdG9yZU9yZGVyfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCBub3QgY2FsbCBvbi1vdXQtb2Ytb3JkZXIgbGlzdGVuZXIgZm9yIHBhY2tldHMgdGhhdCBjb21lIGJlZm9yZSBzeW5jaHJvbml6YXRpb24gc3RhcnQnLCBhc3luYyAoKSA9PiB7XG4gICAgc2FuZGJveC5zdHViKG91dE9mT3JkZXJMaXN0ZW5lciwgJ29uT3V0T2ZPcmRlclBhY2tldCcpLnJldHVybnMoKTtcbiAgICBsZXQgb3V0T2ZPcmRlclBhY2tldCA9IHtcbiAgICAgIGFjY291bnRJZDogJ2FjY291bnRJZCcsXG4gICAgICBzZXF1ZW5jZU51bWJlcjogMTEsXG4gICAgICBwYWNrZXQ6IHt9LFxuICAgICAgcmVjZWl2ZWRBdDogbmV3IERhdGUoJzIwMTAtMTAtMTlUMDk6NTg6NTYuMDAwWicpXG4gICAgfTtcblxuICAgIC8vIFRoZXJlIHdlcmUgbm8gc3luY2hyb25pemF0aW9uIHN0YXJ0IHBhY2tldHNcbiAgICBwYWNrZXRPcmRlcmVyLl9zZXF1ZW5jZU51bWJlckJ5SW5zdGFuY2VbJ2FjY291bnRJZDowJ10gPSB1bmRlZmluZWQ7XG5cbiAgICBwYWNrZXRPcmRlcmVyLl9wYWNrZXRzQnlJbnN0YW5jZVsnYWNjb3VudElkOjAnXSA9IFtvdXRPZk9yZGVyUGFja2V0XTtcbiAgICBhd2FpdCBuZXcgUHJvbWlzZShyZXMgPT4gc2V0VGltZW91dChyZXMsIDEwMDApKTtcbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKG91dE9mT3JkZXJMaXN0ZW5lci5vbk91dE9mT3JkZXJQYWNrZXQpO1xuICB9KS50aW1lb3V0KDMwMDApO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7UGFja2V0T3JkZXJlciNyZXN0b3JlT3JkZXJ9XG4gICAqL1xuICBpdCgnc2hvdWxkIG1haW50YWluIGEgZml4ZWQgcXVldWUgb2Ygd2FpdCBsaXN0JywgKCkgPT4ge1xuICAgIHBhY2tldE9yZGVyZXIuX3dhaXRMaXN0U2l6ZUxpbWl0ID0gMTtcbiAgICBsZXQgc2Vjb25kUGFja2V0ID0ge1xuICAgICAgdHlwZTogJ3ByaWNlcycsXG4gICAgICBzZXF1ZW5jZVRpbWVzdGFtcDogMTYwMzEyNDI2NzE4MCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNCxcbiAgICAgIGFjY291bnRJZDogJ2FjY291bnRJZCcsXG4gICAgICBob3N0OiAncHMtbXBhLTEnXG4gICAgfTtcbiAgICBsZXQgdGhpcmRQYWNrZXQgPSB7XG4gICAgICB0eXBlOiAnYWNjb3VudEluZm9ybWF0aW9uJyxcbiAgICAgIHNlcXVlbmNlVGltZXN0YW1wOiAxNjAzMTI0MjY3MTg3LFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDE1LFxuICAgICAgYWNjb3VudElkOiAnYWNjb3VudElkJyxcbiAgICAgIGhvc3Q6ICdwcy1tcGEtMSdcbiAgICB9O1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHNlY29uZFBhY2tldCk7XG4gICAgcGFja2V0T3JkZXJlci5fcGFja2V0c0J5SW5zdGFuY2VbJ2FjY291bnRJZDowOnBzLW1wYS0xJ10ubGVuZ3RoLnNob3VsZC5lcXVhbCgxKTtcbiAgICBwYWNrZXRPcmRlcmVyLl9wYWNrZXRzQnlJbnN0YW5jZVsnYWNjb3VudElkOjA6cHMtbXBhLTEnXVswXS5wYWNrZXQuc2hvdWxkLmVxdWFsKHNlY29uZFBhY2tldCk7XG4gICAgcGFja2V0T3JkZXJlci5yZXN0b3JlT3JkZXIodGhpcmRQYWNrZXQpO1xuICAgIHBhY2tldE9yZGVyZXIuX3BhY2tldHNCeUluc3RhbmNlWydhY2NvdW50SWQ6MDpwcy1tcGEtMSddLmxlbmd0aC5zaG91bGQuZXF1YWwoMSk7XG4gICAgcGFja2V0T3JkZXJlci5fcGFja2V0c0J5SW5zdGFuY2VbJ2FjY291bnRJZDowOnBzLW1wYS0xJ11bMF0ucGFja2V0LnNob3VsZC5lcXVhbCh0aGlyZFBhY2tldCk7XG4gIH0pO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7UGFja2V0T3JkZXJlciNyZXN0b3JlT3JkZXJ9XG4gICAqL1xuICBpdCgnc2hvdWxkIGNvdW50IHN0YXJ0IHBhY2tldHMgd2l0aCB1bmRlZmluZWQgc3luY2hyb256aWF0aW9uSWQgYXMgb3V0LW9mLW9yZGVyJywgKCkgPT4ge1xuICAgIGxldCBzdGFydFBhY2tldCA9IHtcbiAgICAgIHR5cGU6ICdzeW5jaHJvbml6YXRpb25TdGFydGVkJyxcbiAgICAgIHNlcXVlbmNlVGltZXN0YW1wOiAxNjAzMTI0MjY3MTk4LFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDE2LFxuICAgICAgYWNjb3VudElkOiAnYWNjb3VudElkJyxcbiAgICAgIGhvc3Q6ICdwcy1tcGEtMSdcbiAgICB9O1xuICAgIHBhY2tldE9yZGVyZXIucmVzdG9yZU9yZGVyKHN0YXJ0UGFja2V0KS5zaG91bGQuZGVlcEVxdWFsKFtdKTtcbiAgICBwYWNrZXRPcmRlcmVyLl9wYWNrZXRzQnlJbnN0YW5jZVsnYWNjb3VudElkOjA6cHMtbXBhLTEnXS5sZW5ndGguc2hvdWxkLmVxdWFsKDEpO1xuICAgIHBhY2tldE9yZGVyZXIuX3BhY2tldHNCeUluc3RhbmNlWydhY2NvdW50SWQ6MDpwcy1tcGEtMSddWzBdLnBhY2tldC5zaG91bGQuZGVlcEVxdWFsKHN0YXJ0UGFja2V0KTtcbiAgfSk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtQYWNrZXRPcmRlcmVyI3Jlc3RvcmVPcmRlcn1cbiAgICovXG4gIGl0KCdzaG91bGQgcmVzZXQgc3RhdGUgb24gcmVjb25uZWN0ZWQgZXZlbnQnLCBhc3luYyAoKSA9PiB7XG4gICAgc2FuZGJveC5zdHViKG91dE9mT3JkZXJMaXN0ZW5lciwgJ29uT3V0T2ZPcmRlclBhY2tldCcpLnJldHVybnMoKTtcbiAgICBsZXQgdGltZWRPdXRQYWNrZXQgPSB7XG4gICAgICBhY2NvdW50SWQ6ICdhY2NvdW50SWQnLFxuICAgICAgaW5zdGFuY2VJZDogJ2FjY291bnRJZDowOnBzLW1wYS0xJyxcbiAgICAgIGhvc3Q6ICdwcy1tcGEtMScsXG4gICAgICBpbnN0YW5jZUluZGV4OiAwLFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDExLFxuICAgICAgcGFja2V0OiB7fSxcbiAgICAgIHJlY2VpdmVkQXQ6IG5ldyBEYXRlKCcyMDEwLTEwLTE5VDA5OjU4OjU2LjAwMFonKVxuICAgIH07XG4gICAgbGV0IG5vdFRpbWVkT3V0UGFja2V0ID0ge1xuICAgICAgYWNjb3VudElkOiAnYWNjb3VudElkJyxcbiAgICAgIGluc3RhbmNlSWQ6ICdhY2NvdW50SWQ6MDpwcy1tcGEtMScsXG4gICAgICBob3N0OiAncHMtbXBhLTEnLFxuICAgICAgaW5zdGFuY2VJbmRleDogMCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNSxcbiAgICAgIHBhY2tldDoge30sXG4gICAgICByZWNlaXZlZEF0OiBuZXcgRGF0ZSgnMzAxNS0xMC0xOVQwOTo1ODo1Ni4wMDBaJylcbiAgICB9O1xuICAgIHBhY2tldE9yZGVyZXIuX3NlcXVlbmNlTnVtYmVyQnlJbnN0YW5jZVsnYWNjb3VudElkOjA6cHMtbXBhLTEnXSA9IDE7XG4gICAgcGFja2V0T3JkZXJlci5fcGFja2V0c0J5SW5zdGFuY2VbJ2FjY291bnRJZDowOnBzLW1wYS0xJ10gPSBbXG4gICAgICB0aW1lZE91dFBhY2tldCxcbiAgICAgIG5vdFRpbWVkT3V0UGFja2V0XG4gICAgXTtcbiAgICBwYWNrZXRPcmRlcmVyLm9uUmVjb25uZWN0ZWQoWydhY2NvdW50SWQnXSk7XG4gICAgYXdhaXQgbmV3IFByb21pc2UocmVzID0+IHNldFRpbWVvdXQocmVzLCAxMDAwKSk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChvdXRPZk9yZGVyTGlzdGVuZXIub25PdXRPZk9yZGVyUGFja2V0KTtcbiAgfSk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtQYWNrZXRPcmRlcmVyI3Jlc3RvcmVPcmRlcn1cbiAgICovXG4gIGl0KCdzaG91bGQgcmVzZXQgc3RhdGUgZm9yIGFuIGluc3RhbmNlIG9uIHN0cmVhbSBjbG9zZWQgZXZlbnQnLCBhc3luYyAoKSA9PiB7XG4gICAgc2FuZGJveC5zdHViKG91dE9mT3JkZXJMaXN0ZW5lciwgJ29uT3V0T2ZPcmRlclBhY2tldCcpLnJldHVybnMoKTtcbiAgICBsZXQgdGltZWRPdXRQYWNrZXQgPSB7XG4gICAgICBhY2NvdW50SWQ6ICdhY2NvdW50SWQnLFxuICAgICAgaW5zdGFuY2VJZDogJ2FjY291bnRJZDowOnBzLW1wYS0xJyxcbiAgICAgIGhvc3Q6ICdwcy1tcGEtMScsXG4gICAgICBpbnN0YW5jZUluZGV4OiAwLFxuICAgICAgc2VxdWVuY2VOdW1iZXI6IDExLFxuICAgICAgcGFja2V0OiB7fSxcbiAgICAgIHJlY2VpdmVkQXQ6IG5ldyBEYXRlKCcyMDEwLTEwLTE5VDA5OjU4OjU2LjAwMFonKVxuICAgIH07XG4gICAgbGV0IG5vdFRpbWVkT3V0UGFja2V0ID0ge1xuICAgICAgYWNjb3VudElkOiAnYWNjb3VudElkJyxcbiAgICAgIGluc3RhbmNlSWQ6ICdhY2NvdW50SWQ6MDpwcy1tcGEtMScsXG4gICAgICBob3N0OiAncHMtbXBhLTEnLFxuICAgICAgaW5zdGFuY2VJbmRleDogMCxcbiAgICAgIHNlcXVlbmNlTnVtYmVyOiAxNSxcbiAgICAgIHBhY2tldDoge30sXG4gICAgICByZWNlaXZlZEF0OiBuZXcgRGF0ZSgnMzAxNS0xMC0xOVQwOTo1ODo1Ni4wMDBaJylcbiAgICB9O1xuICAgIHBhY2tldE9yZGVyZXIuX3NlcXVlbmNlTnVtYmVyQnlJbnN0YW5jZVsnYWNjb3VudElkOjA6cHMtbXBhLTEnXSA9IDE7XG4gICAgcGFja2V0T3JkZXJlci5fcGFja2V0c0J5SW5zdGFuY2VbJ2FjY291bnRJZDowOnBzLW1wYS0xJ10gPSBbXG4gICAgICB0aW1lZE91dFBhY2tldCxcbiAgICAgIG5vdFRpbWVkT3V0UGFja2V0XG4gICAgXTtcbiAgICBwYWNrZXRPcmRlcmVyLm9uU3RyZWFtQ2xvc2VkKCdhY2NvdW50SWQ6MDpwcy1tcGEtMScpO1xuICAgIGF3YWl0IG5ldyBQcm9taXNlKHJlcyA9PiBzZXRUaW1lb3V0KHJlcywgMTAwMCkpO1xuICAgIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQob3V0T2ZPcmRlckxpc3RlbmVyLm9uT3V0T2ZPcmRlclBhY2tldCk7XG4gIH0pO1xufSk7XG4iXX0=