'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _assign = require('babel-runtime/core-js/object/assign');

var _assign2 = _interopRequireDefault(_assign);

var _packetLogger = require('./packetLogger.es6');

var _packetLogger2 = _interopRequireDefault(_packetLogger);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {PacketLogger}
 */
describe('PacketLogger', () => {
  let packetLogger, clock, sandbox, packets;
  const folder = './.metaapi/logs/';
  const filePath = folder + '2020-10-10-00/accountId.log';

  function changeSN(obj, sequenceNumber, instanceIndex = 7) {
    return (0, _assign2.default)({}, obj, { sequenceNumber, instanceIndex });
  }

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(async () => {
    clock = _sinon2.default.useFakeTimers({
      now: new Date('2020-10-10 00:00:00.000'),
      shouldAdvanceTime: true
    });
    packetLogger = new _packetLogger2.default({ fileNumberLimit: 3, logFileSizeInHours: 4 });
    packetLogger.start();
    await _fsExtra2.default.emptyDir(folder);
    packets = {
      accountInformation: {
        type: 'accountInformation',
        instanceIndex: 7,
        accountInformation: {
          broker: 'Broker',
          currency: 'USD',
          server: 'Broker-Demo',
          balance: 20000,
          equity: 25000
        },
        sequenceTimestamp: 100000,
        accountId: 'accountId'
      },
      prices: {
        type: 'prices',
        instanceIndex: 7,
        prices: [{
          symbol: 'EURUSD',
          bid: 1.18,
          ask: 1.19
        }, {
          symbol: 'USDJPY',
          bid: 103.222,
          ask: 103.25
        }],
        accountId: 'accountId',
        sequenceTimestamp: 100000,
        sequenceNumber: 1
      },
      status: {
        status: 'connected',
        instanceIndex: 7,
        type: 'status',
        accountId: 'accountId',
        sequenceTimestamp: 100000
      },
      keepalive: {
        instanceIndex: 7,
        type: 'keepalive',
        accountId: 'accountId',
        sequenceTimestamp: 100000
      },
      specifications: {
        specifications: [],
        instanceIndex: 7,
        type: 'specifications',
        accountId: 'accountId',
        sequenceTimestamp: 100000,
        sequenceNumber: 1
      }
    };
  });

  afterEach(async () => {
    await _fsExtra2.default.emptyDir(folder);
    packetLogger.stop();
    clock.restore();
    sandbox.restore();
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record packet', async () => {
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match(JSON.parse(result[0].message), packets.accountInformation);
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should not record status and keepalive packets', async () => {
    packetLogger.logPacket(packets.status);
    packetLogger.logPacket(packets.keepalive);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const exists = await _fsExtra2.default.pathExists(filePath);
    _sinon2.default.assert.match(exists, false);
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record shortened specifications', async () => {
    packetLogger.logPacket(packets.specifications);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match({ type: 'specifications', sequenceNumber: 1, sequenceTimestamp: 100000, instanceIndex: 7 }, JSON.parse(result[0].message));
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record full specifications if compress disabled', async () => {
    packetLogger.stop();
    packetLogger = new _packetLogger2.default({ fileNumberLimit: 3, logFileSizeInHours: 4, compressSpecifications: false });
    packetLogger.start();
    packetLogger.logPacket(packets.specifications);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match({
      accountId: 'accountId',
      type: 'specifications',
      sequenceNumber: 1,
      instanceIndex: 7,
      sequenceTimestamp: 100000,
      specifications: []
    }, JSON.parse(result[0].message));
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record single price packet', async () => {
    packetLogger.logPacket(packets.prices);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match(packets.prices, JSON.parse(result[0].message));
    _sinon2.default.assert.match(packets.accountInformation, JSON.parse(result[1].message));
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record range of price packets', async () => {
    packetLogger.logPacket(packets.prices);
    packetLogger.logPacket(changeSN(packets.prices, 2));
    packetLogger.logPacket(changeSN(packets.prices, 3));
    packetLogger.logPacket(changeSN(packets.prices, 4));
    packetLogger.logPacket(changeSN(packets.keepalive, 5));
    packetLogger.logPacket(changeSN(packets.prices, 6));
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match(packets.prices, JSON.parse(result[0].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 6), JSON.parse(result[1].message));
    _sinon2.default.assert.match('Recorded price packets 1-6, instanceIndex: 7', result[2].message);
    _sinon2.default.assert.match(packets.accountInformation, JSON.parse(result[3].message));
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record range of price packets of different instances', async () => {
    packetLogger.logPacket(packets.prices);
    packetLogger.logPacket(changeSN(packets.prices, 2));
    packetLogger.logPacket(changeSN(packets.prices, 3));
    packetLogger.logPacket(changeSN(packets.prices, 1, 8));
    packetLogger.logPacket(changeSN(packets.prices, 2, 8));
    packetLogger.logPacket(changeSN(packets.prices, 3, 8));
    packetLogger.logPacket(changeSN(packets.prices, 4, 8));
    packetLogger.logPacket(changeSN(packets.prices, 4));
    packetLogger.logPacket(changeSN(packets.prices, 5, 8));
    packetLogger.logPacket((0, _assign2.default)({}, packets.accountInformation, { instanceIndex: 8 }));
    packetLogger.logPacket(changeSN(packets.prices, 5));
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match(packets.prices, JSON.parse(result[0].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 1, 8), JSON.parse(result[1].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 5, 8), JSON.parse(result[2].message));
    _sinon2.default.assert.match('Recorded price packets 1-5, instanceIndex: 8', result[3].message);
    _sinon2.default.assert.match((0, _assign2.default)({}, packets.accountInformation, { instanceIndex: 8 }), JSON.parse(result[4].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 5), JSON.parse(result[5].message));
    _sinon2.default.assert.match('Recorded price packets 1-5, instanceIndex: 7', result[6].message);
    _sinon2.default.assert.match(packets.accountInformation, JSON.parse(result[7].message));
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should record all price packets if compress disabled', async () => {
    packetLogger.stop();
    packetLogger = new _packetLogger2.default({ fileNumberLimit: 3, logFileSizeInHours: 4, compressPrices: false });
    packetLogger.start();
    packetLogger.logPacket(packets.prices);
    packetLogger.logPacket(changeSN(packets.prices, 2));
    packetLogger.logPacket(changeSN(packets.prices, 3));
    packetLogger.logPacket(changeSN(packets.prices, 4));
    packetLogger.logPacket(changeSN(packets.prices, 5));
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 200));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match(packets.prices, JSON.parse(result[0].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 2), JSON.parse(result[1].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 3), JSON.parse(result[2].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 4), JSON.parse(result[3].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 5), JSON.parse(result[4].message));
    _sinon2.default.assert.match(packets.accountInformation, JSON.parse(result[5].message));
  });

  /**
   * @test {PacketLogger#logPacket}
   */
  it('should stop price packet sequence if price sequence number doesnt match', async () => {
    packetLogger.logPacket(packets.prices);
    packetLogger.logPacket(changeSN(packets.prices, 2));
    packetLogger.logPacket(changeSN(packets.prices, 3));
    packetLogger.logPacket(changeSN(packets.prices, 4));
    packetLogger.logPacket(changeSN(packets.prices, 6));
    await clock.tickAsync(1000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId');
    _sinon2.default.assert.match(packets.prices, JSON.parse(result[0].message));
    _sinon2.default.assert.match(changeSN(packets.prices, 4), JSON.parse(result[1].message));
    _sinon2.default.assert.match('Recorded price packets 1-4, instanceIndex: 7', result[2].message);
    _sinon2.default.assert.match(changeSN(packets.prices, 6), JSON.parse(result[3].message));
  });

  /**
   * @test {PacketLogger#readLogs}
   */
  it('should read logs within bounds', async () => {
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(60 * 60 * 1000);
    await new _promise2.default(res => setTimeout(res, 100));
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(60 * 60 * 1000);
    await new _promise2.default(res => setTimeout(res, 100));
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(2000);
    await new _promise2.default(res => setTimeout(res, 100));
    const result = await packetLogger.readLogs('accountId', new Date('2020-10-10 00:30:00.000'), new Date('2020-10-10 01:30:00.000'));
    _sinon2.default.assert.match(result.length, 5);
    const resultAfter = await packetLogger.readLogs('accountId', new Date('2020-10-10 00:30:00.000'));
    _sinon2.default.assert.match(resultAfter.length, 8);
    const resultBefore = await packetLogger.readLogs('accountId', undefined, new Date('2020-10-10 01:30:00.000'));
    _sinon2.default.assert.match(resultBefore.length, 7);
  });

  /**
   * @test {PacketLogger#_deleteOldData}
   */
  it('should delete expired folders', async () => {
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(11000);
    await new _promise2.default(res => setTimeout(res, 100));
    _sinon2.default.assert.match((await _fsExtra2.default.readdir(folder)), ['2020-10-10-00']);

    clock.tick(4 * 60 * 60 * 1000);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(11000);
    await new _promise2.default(res => setTimeout(res, 100));
    _sinon2.default.assert.match((await _fsExtra2.default.readdir(folder)), ['2020-10-10-00', '2020-10-10-01']);

    clock.tick(4 * 60 * 60 * 1000);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(11000);
    await new _promise2.default(res => setTimeout(res, 100));
    _sinon2.default.assert.match((await _fsExtra2.default.readdir(folder)), ['2020-10-10-00', '2020-10-10-01', '2020-10-10-02']);

    clock.tick(4 * 60 * 60 * 1000);
    packetLogger.logPacket(packets.accountInformation);
    await clock.tickAsync(10000);
    await new _promise2.default(res => setTimeout(res, 100));
    _sinon2.default.assert.match((await _fsExtra2.default.readdir(folder)), ['2020-10-10-01', '2020-10-10-02', '2020-10-10-03']);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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