'use strict';

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _metatraderAccount = require('./metatraderAccount.client');

var _metatraderAccount2 = _interopRequireDefault(_metatraderAccount);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const provisioningApiUrl = 'https://mt-provisioning-api-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {MetatraderAccountClient}
 */
describe('MetatraderAccountClient', () => {

  let accountClient;
  const token = 'header.payload.sign';
  let httpClient = new _httpClient2.default();
  let sandbox;
  let requestStub;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    accountClient = new _metatraderAccount2.default(httpClient, token);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {MetatraderAccountClient#getAccounts}
   */
  it('should retrieve MetaTrader accounts from API', async () => {
    let expected = [{
      _id: '1eda642a-a9a3-457c-99af-3bc5e8d5c4c9',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud',
      tags: ['tag1', 'tag2']
    }];
    requestStub.resolves(expected);
    let accounts = await accountClient.getAccounts({
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076'
    });
    accounts.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts`,
      method: 'GET',
      qs: {
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076'
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#getAccounts}
   */
  it('should not retrieve MetaTrader accounts from API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.getAccounts('f9ce1f12-e720-4b9a-9477-c2d4cb25f076');
    } catch (error) {
      error.message.should.equal('You can not invoke getAccounts method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#getAccount}
   */
  it('should retrieve MetaTrader account from API', async () => {
    let expected = {
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud',
      tags: ['tag1', 'tag2']
    };
    requestStub.resolves(expected);
    let account = await accountClient.getAccount('id');
    account.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#getAccountByToken}
   */
  it('should retrieve MetaTrader account by token from API', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    let expected = {
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    };
    requestStub.resolves(expected);
    let account = await accountClient.getAccountByToken();
    account.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/accessToken/token`,
      method: 'GET',
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#createAccount}
   */
  it('should not retrieve MetaTrader account by token via API with api token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, token);
    try {
      await accountClient.getAccountByToken();
    } catch (error) {
      error.message.should.equal('You can not invoke getAccountByToken method, because you have connected with API access token. ' + 'Please use account access token to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#createAccount}
   */
  it('should create MetaTrader account via API', async () => {
    let expected = {
      id: 'id'
    };
    let account = {
      login: '50194988',
      password: 'Test1234',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      type: 'cloud',
      tags: ['tag1']
    };
    requestStub.resolves(expected);
    let id = await accountClient.createAccount(account);
    id.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts`,
      method: 'POST',
      body: account,
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#createAccount}
   */
  it('should not create MetaTrader account via API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.createAccount({});
    } catch (error) {
      error.message.should.equal('You can not invoke createAccount method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#deployAccount}
   */
  it('should deploy MetaTrader account via API', async () => {
    await accountClient.deployAccount('id');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id/deploy`,
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#deployAccount}
   */
  it('should not deploy MetaTrader account via API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.deployAccount('id');
    } catch (error) {
      error.message.should.equal('You can not invoke deployAccount method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#undeployAccount}
   */
  it('should undeploy MetaTrader account via API', async () => {
    await accountClient.undeployAccount('id');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id/undeploy`,
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#undeployAccount}
   */
  it('should not undeploy MetaTrader account via API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.undeployAccount('id');
    } catch (error) {
      error.message.should.equal('You can not invoke undeployAccount method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#redeployAccount}
   */
  it('should redeploy MetaTrader account via API', async () => {
    await accountClient.redeployAccount('id');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id/redeploy`,
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#redeployAccount}
   */
  it('should not redeploy MetaTrader account via API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.redeployAccount('id');
    } catch (error) {
      error.message.should.equal('You can not invoke redeployAccount method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#deleteAccount}
   */
  it('should delete MetaTrader account via API', async () => {
    await accountClient.deleteAccount('id');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id`,
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {MetatraderAccountClient#deleteAccount}
   */
  it('should not delete MetaTrader account from via with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.deleteAccount('id');
    } catch (error) {
      error.message.should.equal('You can not invoke deleteAccount method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#updateAccount}
   */
  it('should update MetaTrader account via API', async () => {
    await accountClient.updateAccount('id', {
      name: 'new account name',
      password: 'new_password007',
      server: 'ICMarketsSC2-Demo',
      tags: ['tag1']
    });
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id`,
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      json: true,
      body: {
        name: 'new account name',
        password: 'new_password007',
        server: 'ICMarketsSC2-Demo',
        tags: ['tag1']
      }
    });
  });

  /**
   * @test {MetatraderAccountClient#updateAccount}
   */
  it('should not update MetaTrader account via API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.updateAccount('id', {});
    } catch (error) {
      error.message.should.equal('You can not invoke updateAccount method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {MetatraderAccountClient#increaseReliability}
   */
  it('should increase MetaTrader account reliability via API', async () => {
    await accountClient.increaseReliability('id');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${provisioningApiUrl}/users/current/accounts/id/increase-reliability`,
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
     * @test {MetatraderAccountClient#increaseReliability}
     */
  it('should not increase MetaTrader account reliability via API with account token', async () => {
    accountClient = new _metatraderAccount2.default(httpClient, 'token');
    try {
      await accountClient.increaseReliability('id');
    } catch (error) {
      error.message.should.equal('You can not invoke increaseReliability method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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