'use strict';

var _assign = require('babel-runtime/core-js/object/assign');

var _assign2 = _interopRequireDefault(_assign);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _metaApiWebsocket = require('./metaApiWebsocket.client');

var _metaApiWebsocket2 = _interopRequireDefault(_metaApiWebsocket);

var _socket = require('socket.io');

var _socket2 = _interopRequireDefault(_socket);

var _notConnectedError = require('./notConnectedError');

var _notConnectedError2 = _interopRequireDefault(_notConnectedError);

var _errorHandler = require('../errorHandler');

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const metaapiApiUrl = 'https://mt-client-api-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {MetaApiWebsocketClient}
 */
// eslint-disable-next-line max-statements
describe('MetaApiWebsocketClient', () => {

  let io;
  let server;
  let client;
  let sandbox;
  let httpClient = new _httpClient2.default();
  const emptyHash = 'd41d8cd98f00b204e9800998ecf8427e';
  const synchronizationThrottler = {
    activeSynchronizationIds: ['synchronizationId'],
    onDisconnect: () => {},
    updateSynchronizationId: () => {}
  };
  let accountInformation = {
    broker: 'True ECN Trading Ltd',
    currency: 'USD',
    server: 'ICMarketsSC-Demo',
    balance: 7319.9,
    equity: 7306.649913200001,
    margin: 184.1,
    freeMargin: 7120.22,
    leverage: 100,
    marginLevel: 3967.58283542
  };

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(async () => {
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: true,
      retryOpts: { retries: 3, minDelayInSeconds: 0.1, maxDelayInSeconds: 0.5 } });
    client.url = 'http://localhost:6784';
    io = new _socket2.default(6784, { path: '/ws', pingTimeout: 1000000 });
    io.on('connect', socket => {
      server = socket;
      if (socket.request._query['auth-token'] !== 'token') {
        socket.emit({ error: 'UnauthorizedError', message: 'Authorization token invalid' });
        socket.close();
      }
    });
    client._socketInstancesByAccounts = { accountId: 0 };
    await client.connect();
    sandbox.stub(client._socketInstances[0].synchronizationThrottler, 'activeSynchronizationIds').get(() => []);
  });

  afterEach(async () => {
    sandbox.restore();
    let resolve;
    let promise = new _promise2.default(res => resolve = res);
    client.close();
    io.close(() => resolve());
    await promise;
  });

  /**
   * @test {MetaApiWebsocketClient#_tryReconnect}
   */
  it('should change client id on reconnect', async () => {
    const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
    client.close();
    let clientId;
    let connectAmount = 0;
    io.on('connect', socket => {
      connectAmount++;
      socket.request.headers['client-id'].should.equal(socket.request._query.clientId);
      socket.request.headers['client-id'].should.not.equal(clientId);
      socket.request._query.clientId.should.not.equal(clientId);
      clientId = socket.request._query.clientId;
      socket.disconnect();
    });
    await client.connect();
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(1500);
    await new _promise2.default(res => setTimeout(res, 50));
    connectAmount.should.be.aboveOrEqual(2);
    clock.restore();
  });

  /**
   * @test {MetaApiWebsocketClient#_getServerUrl}
   */
  it('should connect to dedicated server', async () => {
    let positions = [{
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13
    }];
    let resolve;
    let promise = new _promise2.default(res => resolve = res);
    client.close();
    io.close(() => resolve());
    await promise;
    io = new _socket2.default(6785, { path: '/ws', pingTimeout: 1000000 });
    sandbox.stub(httpClient, 'request').resolves({ url: 'http://localhost:6785' });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: false,
      retryOpts: { retries: 3, minDelayInSeconds: 0.1, maxDelayInSeconds: 0.5 } });
    io.on('connect', socket => {
      server = socket;
      if (socket.request._query['auth-token'] !== 'token') {
        socket.emit({ error: 'UnauthorizedError', message: 'Authorization token invalid' });
        socket.close();
      }
      server.on('request', data => {
        if (data.type === 'getPositions' && data.accountId === 'accountId' && data.application === 'RPC') {
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, positions });
        }
      });
    });
    let actual = await client.getPositions('accountId');
    actual.should.match(positions);
  });

  /**
   * @test {MetaApiWebsocketClient#connect}
   */
  it('should retry connection if first attempt timed out', async () => {
    let positions = [{
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13
    }];
    let resolve;
    let promise = new _promise2.default(res => resolve = res);
    client.close();
    io.close(() => resolve());
    await promise;
    sandbox.stub(httpClient, 'request').resolves({ url: 'http://localhost:6785' });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: false,
      connectTimeout: 0.1,
      retryOpts: { retries: 3, minDelayInSeconds: 0.1, maxDelayInSeconds: 0.5 } });
    (async () => {
      await new _promise2.default(res => setTimeout(res, 200));
      io = new _socket2.default(6785, { path: '/ws', pingTimeout: 30000 });
      io.on('connect', socket => {
        server = socket;
        server.on('request', data => {
          if (data.type === 'getPositions' && data.accountId === 'accountId' && data.application === 'RPC') {
            server.emit('response', { type: 'response', accountId: data.accountId,
              requestId: data.requestId, positions });
          }
        });
      });
    })();
    let actual = await client.getPositions('accountId');
    actual.should.match(positions);
    io.close();
  });

  /**
   * @test {MetaApiWebsocketClient#_getServerUrl}
   */
  it('should retry if region not found', async () => {
    const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
    let resolve;
    let promise = new _promise2.default(res => resolve = res);
    client.close();
    io.close(() => resolve());
    await promise;
    let retryCounter = 0;
    sandbox.stub(httpClient, 'request').callsFake(arg => {
      if (arg.url === 'https://mt-provisioning-api-v1.project-stock.agiliumlabs.cloud/' + 'users/current/regions') {
        retryCounter++;
        if (retryCounter < 3) {
          return ['canada', 'us-west'];
        } else {
          return ['canada', 'us-west', 'germany'];
        }
      }
    });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application', region: 'germany',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: true });
    client._socketInstances = [{
      connected: true,
      requestResolves: [],
      socket: { close: () => {} }
    }];
    clock.tickAsync(5000);
    const url = await client._getServerUrl(0);
    (0, _should2.default)(url).eql('https://mt-client-api-v1.germany.project-stock.agiliumlabs.cloud');
    io.close();
    clock.restore();
  });

  /**
   * @test {MetaApiWebsocketClient#_getServerUrl}
   */
  it('should connect to legacy url if default region selected', async () => {
    client.close();
    sandbox.stub(httpClient, 'request').callsFake(arg => {
      if (arg.url === 'https://mt-provisioning-api-v1.project-stock.agiliumlabs.cloud/' + 'users/current/regions') {
        return ['canada', 'us-west'];
      }
    });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application', region: 'canada',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: true });
    client._socketInstances = [{
      connected: true,
      requestResolves: [],
      socket: { close: () => {} }
    }];
    const url = await client._getServerUrl(0);
    (0, _should2.default)(url).eql('https://mt-client-api-v1.project-stock.agiliumlabs.cloud');
  });

  /**
   * @test {MetaApiWebsocketClient#_getServerUrl}
   */
  it('should connect to shared selected region', async () => {
    client.close();
    sandbox.stub(httpClient, 'request').callsFake(arg => {
      if (arg.url === 'https://mt-provisioning-api-v1.project-stock.agiliumlabs.cloud/' + 'users/current/regions') {
        return ['canada', 'us-west'];
      }
    });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application', region: 'us-west',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: true });
    client._socketInstances = [{
      connected: true,
      requestResolves: [],
      socket: { close: () => {} }
    }];
    const url = await client._getServerUrl(0);
    (0, _should2.default)(url).eql('https://mt-client-api-v1.us-west.project-stock.agiliumlabs.cloud');
  });

  /**
   * @test {MetaApiWebsocketClient#_getServerUrl}
   */
  it('should connect to dedicated selected region', async () => {
    client.close();
    sandbox.stub(httpClient, 'request').callsFake(arg => {
      if (arg.url === 'https://mt-provisioning-api-v1.project-stock.agiliumlabs.cloud/' + 'users/current/regions') {
        return ['canada', 'us-west'];
      } else if (arg.url === 'https://mt-provisioning-api-v1.project-stock.agiliumlabs.cloud/' + 'users/current/servers/mt-client-api') {
        return {
          url: 'http://localhost:8081',
          hostname: 'mt-client-api-dedicated',
          domain: 'project-stock.agiliumlabs.cloud'
        };
      }
    });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application', region: 'us-west',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: true });
    client._socketInstances = [{
      connected: true,
      requestResolves: [],
      socket: { close: () => {} }
    }];
    const url = await client._getServerUrl(0);
    (0, _should2.default)(url).eql('https://mt-client-api-v1.us-west.project-stock.agiliumlabs.cloud');
  });

  /**
   * @test {MetaApiWebsocketClient#getAccountInformation}
   */
  it('should retrieve MetaTrader account information from API', async () => {
    server.on('request', data => {
      if (data.type === 'getAccountInformation' && data.accountId === 'accountId' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId,
          accountInformation
        });
      }
    });
    let actual = await client.getAccountInformation('accountId');
    actual.should.match(accountInformation);
  });

  /**
   * @test {MetaApiWebsocketClient#getPositions}
   */
  it('should retrieve MetaTrader positions from API', async () => {
    let positions = [{
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13
    }];
    server.on('request', data => {
      if (data.type === 'getPositions' && data.accountId === 'accountId' && data.application === 'RPC') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, positions });
      }
    });
    let actual = await client.getPositions('accountId');
    actual.should.match(positions);
  });

  /**
   * @test {MetaApiWebsocketClient#getPosition}
   */
  it('should retrieve MetaTrader position from API by id', async () => {
    let position = {
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13
    };
    server.on('request', data => {
      if (data.type === 'getPosition' && data.accountId === 'accountId' && data.positionId === '46214692' && data.application === 'RPC') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, position });
      }
    });
    let actual = await client.getPosition('accountId', '46214692');
    actual.should.match(position);
  });

  /**
   * @test {MetaApiWebsocketClient#getOrders}
   */
  it('should retrieve MetaTrader orders from API', async () => {
    let orders = [{
      id: '46871284',
      type: 'ORDER_TYPE_BUY_LIMIT',
      state: 'ORDER_STATE_PLACED',
      symbol: 'AUDNZD',
      magic: 123456,
      platform: 'mt5',
      time: new Date('2020-04-20T08:38:58.270Z'),
      openPrice: 1.03,
      currentPrice: 1.05206,
      volume: 0.01,
      currentVolume: 0.01,
      comment: 'COMMENT2'
    }];
    server.on('request', data => {
      if (data.type === 'getOrders' && data.accountId === 'accountId' && data.application === 'RPC') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, orders });
      }
    });
    let actual = await client.getOrders('accountId');
    actual.should.match(orders);
  });

  /**
   * @test {MetaApiWebsocketClient#getOrder}
   */
  it('should retrieve MetaTrader order from API by id', async () => {
    let order = {
      id: '46871284',
      type: 'ORDER_TYPE_BUY_LIMIT',
      state: 'ORDER_STATE_PLACED',
      symbol: 'AUDNZD',
      magic: 123456,
      platform: 'mt5',
      time: new Date('2020-04-20T08:38:58.270Z'),
      openPrice: 1.03,
      currentPrice: 1.05206,
      volume: 0.01,
      currentVolume: 0.01,
      comment: 'COMMENT2'
    };
    server.on('request', data => {
      if (data.type === 'getOrder' && data.accountId === 'accountId' && data.orderId === '46871284' && data.application === 'RPC') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, order });
      }
    });
    let actual = await client.getOrder('accountId', '46871284');
    actual.should.match(order);
  });

  /**
   * @test {MetaApiWebsocketClient#getHistoryOrdersByTicket}
   */
  it('should retrieve MetaTrader history orders from API by ticket', async () => {
    let historyOrders = [{
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      currentPrice: 1.261,
      currentVolume: 0,
      doneTime: new Date('2020-04-15T02:45:06.521Z'),
      id: '46214692',
      magic: 1000,
      platform: 'mt5',
      positionId: '46214692',
      state: 'ORDER_STATE_FILLED',
      symbol: 'GBPUSD',
      time: new Date('2020-04-15T02:45:06.260Z'),
      type: 'ORDER_TYPE_BUY',
      volume: 0.07
    }];
    server.on('request', data => {
      if (data.type === 'getHistoryOrdersByTicket' && data.accountId === 'accountId' && data.ticket === '46214692' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, historyOrders,
          synchronizing: false
        });
      }
    });
    let actual = await client.getHistoryOrdersByTicket('accountId', '46214692');
    actual.should.match({ historyOrders, synchronizing: false });
  });

  /**
   * @test {MetaApiWebsocketClient#getHistoryOrdersByPosition}
   */
  it('should retrieve MetaTrader history orders from API by position', async () => {
    let historyOrders = [{
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      currentPrice: 1.261,
      currentVolume: 0,
      doneTime: new Date('2020-04-15T02:45:06.521Z'),
      id: '46214692',
      magic: 1000,
      platform: 'mt5',
      positionId: '46214692',
      state: 'ORDER_STATE_FILLED',
      symbol: 'GBPUSD',
      time: new Date('2020-04-15T02:45:06.260Z'),
      type: 'ORDER_TYPE_BUY',
      volume: 0.07
    }];
    server.on('request', data => {
      if (data.type === 'getHistoryOrdersByPosition' && data.accountId === 'accountId' && data.positionId === '46214692' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, historyOrders,
          synchronizing: false
        });
      }
    });
    let actual = await client.getHistoryOrdersByPosition('accountId', '46214692');
    actual.should.match({ historyOrders, synchronizing: false });
  });

  /**
   * @test {MetaApiWebsocketClient#getHistoryOrdersByTimeRange}
   */
  it('should retrieve MetaTrader history orders from API by time range', async () => {
    let historyOrders = [{
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      currentPrice: 1.261,
      currentVolume: 0,
      doneTime: new Date('2020-04-15T02:45:06.521Z'),
      id: '46214692',
      magic: 1000,
      platform: 'mt5',
      positionId: '46214692',
      state: 'ORDER_STATE_FILLED',
      symbol: 'GBPUSD',
      time: new Date('2020-04-15T02:45:06.260Z'),
      type: 'ORDER_TYPE_BUY',
      volume: 0.07
    }];
    server.on('request', data => {
      if (data.type === 'getHistoryOrdersByTimeRange' && data.accountId === 'accountId' && data.startTime === '2020-04-15T02:45:00.000Z' && data.endTime === '2020-04-15T02:46:00.000Z' && data.offset === 1 && data.limit === 100 && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, historyOrders,
          synchronizing: false
        });
      }
    });
    let actual = await client.getHistoryOrdersByTimeRange('accountId', new Date('2020-04-15T02:45:00.000Z'), new Date('2020-04-15T02:46:00.000Z'), 1, 100);
    actual.should.match({ historyOrders, synchronizing: false });
  });

  /**
   * @test {MetaApiWebsocketClient#getDealsByTicket}
   */
  it('should retrieve MetaTrader deals from API by ticket', async () => {
    let deals = [{
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      commission: -0.25,
      entryType: 'DEAL_ENTRY_IN',
      id: '33230099',
      magic: 1000,
      platform: 'mt5',
      orderId: '46214692',
      positionId: '46214692',
      price: 1.26101,
      profit: 0,
      swap: 0,
      symbol: 'GBPUSD',
      time: new Date('2020-04-15T02:45:06.521Z'),
      type: 'DEAL_TYPE_BUY',
      volume: 0.07
    }];
    server.on('request', data => {
      if (data.type === 'getDealsByTicket' && data.accountId === 'accountId' && data.ticket === '46214692' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, deals,
          synchronizing: false
        });
      }
    });
    let actual = await client.getDealsByTicket('accountId', '46214692');
    actual.should.match({ deals, synchronizing: false });
  });

  /**
   * @test {MetaApiWebsocketClient#getDealsByPosition}
   */
  it('should retrieve MetaTrader deals from API by position', async () => {
    let deals = [{
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      commission: -0.25,
      entryType: 'DEAL_ENTRY_IN',
      id: '33230099',
      magic: 1000,
      platform: 'mt5',
      orderId: '46214692',
      positionId: '46214692',
      price: 1.26101,
      profit: 0,
      swap: 0,
      symbol: 'GBPUSD',
      time: new Date('2020-04-15T02:45:06.521Z'),
      type: 'DEAL_TYPE_BUY',
      volume: 0.07
    }];
    server.on('request', data => {
      if (data.type === 'getDealsByPosition' && data.accountId === 'accountId' && data.positionId === '46214692' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, deals,
          synchronizing: false
        });
      }
    });
    let actual = await client.getDealsByPosition('accountId', '46214692');
    actual.should.match({ deals, synchronizing: false });
  });

  /**
   * @test {MetaApiWebsocketClient#getDealsByTimeRange}
   */
  it('should retrieve MetaTrader deals from API by time range', async () => {
    let deals = [{
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      commission: -0.25,
      entryType: 'DEAL_ENTRY_IN',
      id: '33230099',
      magic: 1000,
      platform: 'mt5',
      orderId: '46214692',
      positionId: '46214692',
      price: 1.26101,
      profit: 0,
      swap: 0,
      symbol: 'GBPUSD',
      time: new Date('2020-04-15T02:45:06.521Z'),
      type: 'DEAL_TYPE_BUY',
      volume: 0.07
    }];
    server.on('request', data => {
      if (data.type === 'getDealsByTimeRange' && data.accountId === 'accountId' && data.startTime === '2020-04-15T02:45:00.000Z' && data.endTime === '2020-04-15T02:46:00.000Z' && data.offset === 1 && data.limit === 100 && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, deals,
          synchronizing: false
        });
      }
    });
    let actual = await client.getDealsByTimeRange('accountId', new Date('2020-04-15T02:45:00.000Z'), new Date('2020-04-15T02:46:00.000Z'), 1, 100);
    actual.should.match({ deals, synchronizing: false });
  });

  /**
   * @test {MetaApiWebsocketClient#removeHistory}
   */
  it('should remove history from API', async () => {
    let requestReceived = false;
    server.on('request', data => {
      if (data.type === 'removeHistory' && data.accountId === 'accountId' && data.application === 'app') {
        requestReceived = true;
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
      }
    });
    await client.removeHistory('accountId', 'app');
    requestReceived.should.be.true();
  });

  /**
   * @test {MetaApiWebsocketClient#removeApplication}
   */
  it('should remove application from API', async () => {
    let requestReceived = false;
    server.on('request', data => {
      if (data.type === 'removeApplication' && data.accountId === 'accountId' && data.application === 'application') {
        requestReceived = true;
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
      }
    });
    await client.removeApplication('accountId');
    requestReceived.should.be.true();
  });

  /**
   * @test {MetaApiWebsocketClient#trade}
   */
  it('should execute a trade via new API version', async () => {
    let trade = {
      actionType: 'ORDER_TYPE_SELL',
      symbol: 'AUDNZD',
      volume: 0.07
    };
    let response = {
      numericCode: 10009,
      stringCode: 'TRADE_RETCODE_DONE',
      message: 'Request completed',
      orderId: '46870472'
    };
    server.on('request', data => {
      data.trade.should.match(trade);
      if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, response });
      }
    });
    let actual = await client.trade('accountId', trade);
    actual.should.match(response);
  });

  /**
   * @test {MetaApiWebsocketClient#trade}
   */
  it('should execute a trade via API and receive trade error from old API version', async () => {
    let trade = {
      actionType: 'ORDER_TYPE_SELL',
      symbol: 'AUDNZD',
      volume: 0.07
    };
    let response = {
      error: 10006,
      description: 'TRADE_RETCODE_REJECT',
      message: 'Request rejected',
      orderId: '46870472'
    };
    server.on('request', data => {
      data.trade.should.match(trade);
      if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, response });
      }
    });
    try {
      await client.trade('accountId', trade);
      _should2.default.fail('Trade error expected');
    } catch (err) {
      err.message.should.equal('Request rejected');
      err.name.should.equal('TradeError');
      err.stringCode.should.equal('TRADE_RETCODE_REJECT');
      err.numericCode.should.equal(10006);
    }
  });

  /**
   * @test {MetaApiWebsocketClient#subscribe}
   */
  it('should connect to MetaTrader terminal', async () => {
    let requestReceived = false;
    server.on('request', data => {
      if (data.type === 'subscribe' && data.accountId === 'accountId' && data.application === 'application' && data.instanceIndex === 1) {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        requestReceived = true;
      }
    });
    await client.subscribe('accountId', 1);
    await new _promise2.default(res => setTimeout(res, 50));
    requestReceived.should.be.true();
  });

  /**
   * @test {MetaApiWebsocketClient#subscribe}
   */
  it('should create new instance when account limit is reached', async () => {
    _sinon2.default.assert.match(client.socketInstances.length, 1);
    for (let i = 0; i < 100; i++) {
      client._socketInstancesByAccounts['accountId' + i] = 0;
    }

    io.removeAllListeners('connect');
    io.on('connect', socket => {
      socket.on('request', data => {
        if (data.type === 'subscribe' && data.accountId === 'accountId101' && data.application === 'application' && data.instanceIndex === 1) {
          socket.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        }
      });
    });
    await client.subscribe('accountId101', 1);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.match(client.socketInstances.length, 2);
  });

  /**
   * @test {MetaApiWebsocketClient#subscribe}
   */
  it('should return error if connect to MetaTrader terminal failed', async () => {
    let requestReceived = false;
    server.on('request', data => {
      if (data.type === 'subscribe' && data.accountId === 'accountId' && data.application === 'application') {
        requestReceived = true;
      }
      server.emit('processingError', {
        id: 1, error: 'NotAuthenticatedError', message: 'Error message',
        requestId: data.requestId
      });
    });
    let success = true;
    try {
      await client.subscribe('accountId');
      success = false;
    } catch (err) {
      err.name.should.equal('NotConnectedError');
    }
    success.should.be.true();
    requestReceived.should.be.true();
  });

  /**
   * @test {MetaApiWebsocketClient#getSymbols}
   */
  it('should retrieve symbols from API', async () => {
    let symbols = ['EURUSD'];
    server.on('request', data => {
      if (data.type === 'getSymbols' && data.accountId === 'accountId' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId, symbols
        });
      }
    });
    let actual = await client.getSymbols('accountId');
    actual.should.match(symbols);
  });

  /**
   * @test {MetaApiWebsocketClient#getSymbolSpecification}
   */
  it('should retrieve symbol specification from API', async () => {
    let specification = {
      symbol: 'AUDNZD',
      tickSize: 0.00001,
      minVolume: 0.01,
      maxVolume: 100,
      volumeStep: 0.01
    };
    server.on('request', data => {
      if (data.type === 'getSymbolSpecification' && data.accountId === 'accountId' && data.symbol === 'AUDNZD' && data.application === 'RPC') {
        server.emit('response', {
          type: 'response', accountId: data.accountId, requestId: data.requestId,
          specification
        });
      }
    });
    let actual = await client.getSymbolSpecification('accountId', 'AUDNZD');
    actual.should.match(specification);
  });

  /**
   * @test {MetaApiWebsocketClient#getSymbolPrice}
   */
  it('should retrieve symbol price from API', async () => {
    let price = {
      symbol: 'AUDNZD',
      bid: 1.05297,
      ask: 1.05309,
      profitTickValue: 0.59731,
      lossTickValue: 0.59736
    };
    server.on('request', data => {
      if (data.type === 'getSymbolPrice' && data.accountId === 'accountId' && data.symbol === 'AUDNZD' && data.application === 'RPC' && data.keepSubscription === true) {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, price });
      }
    });
    let actual = await client.getSymbolPrice('accountId', 'AUDNZD', true);
    actual.should.match(price);
  });

  /**
   * @test {MetaApiWebsocketClient#getCandle}
   */
  it('should retrieve candle from API', async () => {
    let candle = {
      symbol: 'AUDNZD',
      timeframe: '15m',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      open: 1.03297,
      high: 1.06309,
      low: 1.02705,
      close: 1.043,
      tickVolume: 1435,
      spread: 17,
      volume: 345
    };
    server.on('request', data => {
      if (data.type === 'getCandle' && data.accountId === 'accountId' && data.symbol === 'AUDNZD' && data.application === 'RPC' && data.timeframe === '15m' && data.keepSubscription === true) {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, candle });
      }
    });
    let actual = await client.getCandle('accountId', 'AUDNZD', '15m', true);
    actual.should.match(candle);
  });

  /**
   * @test {MetaApiWebsocketClient#getTick}
   */
  it('should retrieve latest tick from API', async () => {
    let tick = {
      symbol: 'AUDNZD',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      bid: 1.05297,
      ask: 1.05309,
      last: 0.5298,
      volume: 0.13,
      side: 'buy'
    };
    server.on('request', data => {
      if (data.type === 'getTick' && data.accountId === 'accountId' && data.symbol === 'AUDNZD' && data.application === 'RPC' && data.keepSubscription === true) {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, tick });
      }
    });
    let actual = await client.getTick('accountId', 'AUDNZD', true);
    actual.should.match(tick);
  });

  /**
   * @test {MetaApiWebsocketClient#getBook}
   */
  it('should retrieve latest order book from API', async () => {
    let book = {
      symbol: 'AUDNZD',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      book: [{
        type: 'BOOK_TYPE_SELL',
        price: 1.05309,
        volume: 5.67
      }, {
        type: 'BOOK_TYPE_BUY',
        price: 1.05297,
        volume: 3.45
      }]
    };
    server.on('request', data => {
      if (data.type === 'getBook' && data.accountId === 'accountId' && data.symbol === 'AUDNZD' && data.application === 'RPC' && data.keepSubscription === true) {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, book });
      }
    });
    let actual = await client.getBook('accountId', 'AUDNZD', true);
    actual.should.match(book);
  });

  /**
   * @test {MetaApiWebsocketClient#sendUptime}
   */
  it('should sent uptime stats to the server', async () => {
    server.on('request', data => {
      if (data.type === 'saveUptime' && data.accountId === 'accountId' && (0, _stringify2.default)(data.uptime) === (0, _stringify2.default)({ '1h': 100 }) && data.application === 'application') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
      }
    });
    await client.saveUptime('accountId', { '1h': 100 });
  });

  /**
   * @test {MetaApiWebsocketClient#unsubscribe}
   */
  describe('unsubscription', () => {

    /**
    * @test {MetaApiWebsocketClient#unsubscribe}
    */
    it('should unsubscribe from account data', async () => {
      let requestReceived = false;

      let response = { type: 'response', accountId: 'accountId' };
      server.on('request', data => {
        if (data.type === 'unsubscribe' && data.accountId === 'accountId') {
          requestReceived = true;
          server.emit('response', (0, _assign2.default)({ requestId: data.requestId }, response));
        }
      });
      await client.unsubscribe('accountId');
      _sinon2.default.assert.match(requestReceived, true);
      client.socketInstancesByAccounts.should.not.have.property('accountId');
    });

    /**
     * @test {MetaApiWebsocketClient#unsubscribe}
     */
    it('should ignore not found exception on unsubscribe', async () => {
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'ValidationError', message: 'Validation failed',
          details: [{ parameter: 'volume', message: 'Required value.' }], requestId: data.requestId
        });
      });
      try {
        await client.unsubscribe('accountId');
        throw new Error('ValidationError extected');
      } catch (err) {
        err.name.should.equal('ValidationError');
        err.details.should.match([{
          parameter: 'volume',
          message: 'Required value.'
        }]);
      }
      server.removeAllListeners('request');
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'NotFoundError', message: 'Account not found', requestId: data.requestId
        });
      });
      await client.unsubscribe('accountId');
    });

    /**
     * @test {MetaApiWebsocketClient#unsubscribe}
     */
    it('should ignore timeout error on unsubscribe', async () => {
      let clock = sandbox.useFakeTimers({ shouldAdvanceTime: true });
      let promise = client.unsubscribe('accountId').catch(() => {});
      await clock.tickAsync(15000);
      await promise;
    }).timeout(20000);

    /**
     * @test {MetaApiWebsocketClient#unsubscribe}
     */
    it('should repeat unsubscription on synchronization packets if account must be unsubscribed', async () => {
      let subscribeServerHandler = sandbox.stub();
      let unsubscribeServerHandler = sandbox.stub();
      let clock = sandbox.useFakeTimers({ shouldAdvanceTime: true });
      server.on('request', data => {
        let serverHandler;
        if (data.type === 'subscribe' && data.accountId === 'accountId') {
          serverHandler = subscribeServerHandler;
        } else if (data.type === 'unsubscribe' && data.accountId === 'accountId') {
          serverHandler = unsubscribeServerHandler;
        }
        if (serverHandler) {
          serverHandler();
          let response = { type: 'response', accountId: 'accountId' };
          server.emit('response', (0, _assign2.default)({ requestId: data.requestId }, response));
        }
      });
      // Subscribing
      await client.subscribe('accountId', 1);
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledOnce(subscribeServerHandler);
      // Unsubscribing
      await client.unsubscribe('accountId');
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledOnce(unsubscribeServerHandler);
      // Sending a packet, should throttle first repeat unsubscribe request
      server.emit('synchronization', {
        type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1
      });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledOnce(unsubscribeServerHandler);
      // Repeat a packet after a while, should unsubscribe again
      await clock.tickAsync(11000);
      server.emit('synchronization', {
        type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1
      });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledTwice(unsubscribeServerHandler);
      // Repeat a packet, should throttle unsubscribe request
      server.emit('synchronization', {
        type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1
      });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledTwice(unsubscribeServerHandler);
      // Repeat a packet after a while, should not throttle unsubscribe request
      await clock.tickAsync(11000);
      server.emit('synchronization', {
        type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1
      });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledThrice(unsubscribeServerHandler);
    });
  });

  describe('error handling', () => {

    /**
     * @test {MetaApiWebsocketClient#trade}
     */
    it('should handle ValidationError', async () => {
      let trade = {
        actionType: 'ORDER_TYPE_SELL',
        symbol: 'AUDNZD'
      };
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'ValidationError', message: 'Validation failed',
          details: [{ parameter: 'volume', message: 'Required value.' }], requestId: data.requestId
        });
      });
      try {
        await client.trade('accountId', trade);
        throw new Error('ValidationError extected');
      } catch (err) {
        err.name.should.equal('ValidationError');
        err.details.should.match([{
          parameter: 'volume',
          message: 'Required value.'
        }]);
      }
    });

    /**
     * @test {MetaApiWebsocketClient#getPosition}
     */
    it('should handle NotFoundError', async () => {
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'NotFoundError', message: 'Position id 1234 not found',
          requestId: data.requestId
        });
      });
      try {
        await client.getPosition('accountId', '1234');
        throw new Error('NotFoundError extected');
      } catch (err) {
        err.name.should.equal('NotFoundError');
      }
    });

    /**
     * @test {MetaApiWebsocketClient#getPosition}
     */
    it('should handle NotSynchronizedError', async () => {
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'NotSynchronizedError', message: 'Error message',
          requestId: data.requestId
        });
      });
      try {
        await client.getPosition('accountId', '1234');
        throw new Error('NotSynchronizedError extected');
      } catch (err) {
        err.name.should.equal('NotSynchronizedError');
      }
    }).timeout(8000);

    /**
     * @test {MetaApiWebsocketClient#getPosition}
     */
    it('should handle NotConnectedError', async () => {
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'NotAuthenticatedError', message: 'Error message',
          requestId: data.requestId
        });
      });
      try {
        await client.getPosition('accountId', '1234');
        throw new Error('NotConnectedError extected');
      } catch (err) {
        err.name.should.equal('NotConnectedError');
      }
    });

    /**
     * @test {MetaApiWebsocketClient#getPosition}
     */
    it('should handle other errors', async () => {
      server.on('request', data => {
        server.emit('processingError', {
          id: 1, error: 'Error', message: 'Error message',
          requestId: data.requestId
        });
      });
      try {
        await client.getPosition('accountId', '1234');
        throw new Error('InternalError extected');
      } catch (err) {
        err.name.should.equal('InternalError');
      }
    }).timeout(8000);
  });

  describe('connection status synchronization', () => {

    beforeEach(() => {
      sandbox.stub(client._subscriptionManager, 'isSubscriptionActive').returns(true);
    });

    afterEach(() => {
      client.removeAllListeners();
    });

    it('should process authenticated synchronization event', async () => {
      let listener = {
        onConnected: () => {}
      };
      sandbox.stub(listener, 'onConnected').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1, replicas: 2 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onConnected, '1:ps-mpa-1', 2);
    });

    it('should process authenticated synchronization event with session id', async () => {
      let listener = {
        onConnected: () => {}
      };
      sandbox.stub(listener, 'onConnected').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 2, replicas: 4, sessionId: 'wrong' });
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1, replicas: 2, sessionId: client._sessionId });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.callCount(listener.onConnected, 1);
      _sinon2.default.assert.calledWith(listener.onConnected, '1:ps-mpa-1', 2);
    });

    it('should process broker connection status event', async () => {
      let listener = {
        onConnected: () => {},
        onBrokerConnectionStatusChanged: () => {}
      };
      sandbox.stub(listener, 'onBrokerConnectionStatusChanged').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onBrokerConnectionStatusChanged, '1:ps-mpa-1', true);
    });

    it('should call an onDisconnect if there was no signal for a long time', async () => {
      const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
      let listener = {
        onConnected: () => {},
        onDisconnected: () => {},
        onBrokerConnectionStatusChanged: () => {}
      };
      sandbox.stub(listener, 'onDisconnected').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1, replicas: 2 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(10000);
      await server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(55000);
      _sinon2.default.assert.notCalled(listener.onDisconnected);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1, replicas: 2 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(10000);
      _sinon2.default.assert.notCalled(listener.onDisconnected);
      await clock.tickAsync(55000);
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onDisconnected, '1:ps-mpa-1');
      await clock.tickAsync(10000);
      clock.restore();
    });

    it('should close stream on timeout if another stream exists', async () => {
      const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
      let listener = {
        onConnected: () => {},
        onDisconnected: () => {},
        onStreamClosed: () => {},
        onBrokerConnectionStatusChanged: () => {}
      };
      const onTimeoutStub = sandbox.stub(client._subscriptionManager, 'onTimeout').resolves();
      const onStreamClosedStub = sandbox.stub(listener, 'onStreamClosed').resolves();
      const onDisconnectedStub = sandbox.stub(listener, 'onDisconnected').resolves();
      sandbox.stub(client._subscriptionManager, 'onDisconnected').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1, replicas: 2 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(15000);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-2',
        instanceIndex: 1, replicas: 2 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-2', connected: true,
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(15000);
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-2', connected: true,
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(55000);
      _sinon2.default.assert.notCalled(onDisconnectedStub);
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        instanceIndex: 1 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-2', connected: true,
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(15000);
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-2', connected: true,
        instanceIndex: 1 });
      _sinon2.default.assert.notCalled(onDisconnectedStub);
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(55000);
      _sinon2.default.assert.calledWith(onStreamClosedStub, '1:ps-mpa-1');
      _sinon2.default.assert.notCalled(onDisconnectedStub);
      _sinon2.default.assert.notCalled(onTimeoutStub);
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(15000);
      _sinon2.default.assert.calledWith(onDisconnectedStub, '1:ps-mpa-2');
      _sinon2.default.assert.notCalled(client._subscriptionManager.onDisconnected);
      _sinon2.default.assert.calledWith(onTimeoutStub, 'accountId', 1);
      clock.restore();
    });

    it('should process server-side health status event', async () => {
      let listener = {
        onConnected: () => {},
        onBrokerConnectionStatusChanged: () => {},
        onHealthStatus: () => {}
      };
      sandbox.stub(listener, 'onHealthStatus').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1 });
      server.emit('synchronization', { type: 'status', accountId: 'accountId', host: 'ps-mpa-1', connected: true,
        healthStatus: { restApiHealthy: true }, instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onHealthStatus, '1:ps-mpa-1', { restApiHealthy: true });
    });

    it('should process disconnected synchronization event', async () => {
      let listener = {
        onConnected: () => {},
        onDisconnected: () => {}
      };
      sandbox.stub(listener, 'onDisconnected').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1 });
      server.emit('synchronization', { type: 'disconnected', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onDisconnected, '1:ps-mpa-1');
    });

    it('should close the stream if host name disconnected and another stream exists', async () => {
      let listener = {
        onConnected: () => {},
        onDisconnected: () => {},
        onStreamClosed: () => {}
      };
      sandbox.stub(listener, 'onDisconnected').resolves();
      sandbox.stub(listener, 'onStreamClosed').resolves();
      const onDisconnectedStub = sandbox.stub(client._subscriptionManager, 'onDisconnected').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1, replicas: 2 });
      server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-2',
        instanceIndex: 1, replicas: 2 });
      server.emit('synchronization', { type: 'disconnected', accountId: 'accountId', host: 'ps-mpa-1',
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onStreamClosed, '1:ps-mpa-1');
      _sinon2.default.assert.notCalled(listener.onDisconnected);
      _sinon2.default.assert.notCalled(onDisconnectedStub);
      server.emit('synchronization', { type: 'disconnected', accountId: 'accountId', host: 'ps-mpa-2',
        instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledOnce(listener.onDisconnected);
      _sinon2.default.assert.calledWith(onDisconnectedStub, 'accountId', 1);
    });
  });

  describe('terminal state synchronization', () => {

    beforeEach(() => {
      sandbox.stub(client._subscriptionManager, 'isSubscriptionActive').returns(true);
    });

    afterEach(() => {
      client.removeAllListeners();
    });

    it('should only accept packets with own synchronization ids', async () => {
      let listener = {
        onAccountInformationUpdated: () => {}
      };
      sandbox.stub(listener, 'onAccountInformationUpdated').resolves();
      client.addSynchronizationListener('accountId', listener);
      sandbox.stub(client._socketInstances[0].synchronizationThrottler, 'activeSynchronizationIds').get(() => ['synchronizationId']);
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation: {}, instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.callCount(listener.onAccountInformationUpdated, 1);
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation: {}, instanceIndex: 1, synchronizationId: 'wrong' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.callCount(listener.onAccountInformationUpdated, 1);
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation: {}, instanceIndex: 1, synchronizationId: 'synchronizationId' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.callCount(listener.onAccountInformationUpdated, 2);
    });

    /**
     * @test {MetaApiWebsocketClient#synchronize}
     */
    it('should synchronize with MetaTrader terminal', async () => {
      let requestReceived = false;
      // eslint-disable-next-line complexity
      server.on('request', data => {
        if (data.type === 'synchronize' && data.accountId === 'accountId' && data.host === 'ps-mpa-1' && data.startingHistoryOrderTime === '2020-01-01T00:00:00.000Z' && data.startingDealTime === '2020-01-02T00:00:00.000Z' && data.requestId === 'synchronizationId' && data.application === 'application' && data.instanceIndex === 1) {
          requestReceived = true;
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        }
      });
      await client.synchronize('accountId', 1, 'ps-mpa-1', 'synchronizationId', new Date('2020-01-01T00:00:00.000Z'), new Date('2020-01-02T00:00:00.000Z'), emptyHash, emptyHash, emptyHash);
      requestReceived.should.be.true();
    });

    it('should process synchronization started event', async () => {
      client._socketInstances[0].synchronizationThrottler = synchronizationThrottler;
      let listener = {
        onSynchronizationStarted: () => {},
        onPositionsSynchronized: () => {},
        onPendingOrdersSynchronized: () => {},
        onAccountInformationUpdated: () => {}
      };
      sandbox.stub(listener, 'onSynchronizationStarted').resolves();
      sandbox.stub(listener, 'onPositionsSynchronized').resolves();
      sandbox.stub(listener, 'onPendingOrdersSynchronized').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'synchronizationStarted', accountId: 'accountId', instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1' });
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation, instanceIndex: 1, host: 'ps-mpa-1', synchronizationId: 'synchronizationId' });
      await new _promise2.default(res => setTimeout(res, 100));
      _sinon2.default.assert.calledWith(listener.onSynchronizationStarted, '1:ps-mpa-1', true, true, true);
      _sinon2.default.assert.notCalled(listener.onPositionsSynchronized);
      _sinon2.default.assert.notCalled(listener.onPendingOrdersSynchronized);
    });

    it('should process synchronization started event with no updates', async () => {
      client._socketInstances[0].synchronizationThrottler = synchronizationThrottler;
      let listener = {
        onSynchronizationStarted: () => {},
        onPositionsSynchronized: () => {},
        onPendingOrdersSynchronized: () => {},
        onAccountInformationUpdated: () => {}
      };
      sandbox.stub(listener, 'onSynchronizationStarted').resolves();
      sandbox.stub(listener, 'onPositionsSynchronized').resolves();
      sandbox.stub(listener, 'onPendingOrdersSynchronized').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'synchronizationStarted', accountId: 'accountId', instanceIndex: 1,
        specificationsUpdated: false, positionsUpdated: false, ordersUpdated: false,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1' });
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation, instanceIndex: 1, host: 'ps-mpa-1', synchronizationId: 'synchronizationId' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onSynchronizationStarted, '1:ps-mpa-1', false, false, false);
      _sinon2.default.assert.calledWith(listener.onPositionsSynchronized, '1:ps-mpa-1', 'synchronizationId');
      _sinon2.default.assert.calledWith(listener.onPendingOrdersSynchronized, '1:ps-mpa-1', 'synchronizationId');
    });

    it('should process synchronization started event without updating positions', async () => {
      let orders = [{
        id: '46871284',
        type: 'ORDER_TYPE_BUY_LIMIT',
        state: 'ORDER_STATE_PLACED',
        symbol: 'AUDNZD',
        magic: 123456,
        platform: 'mt5',
        time: new Date('2020-04-20T08:38:58.270Z'),
        openPrice: 1.03,
        currentPrice: 1.05206,
        volume: 0.01,
        currentVolume: 0.01,
        comment: 'COMMENT2'
      }];
      client._socketInstances[0].synchronizationThrottler = synchronizationThrottler;
      let listener = {
        onSynchronizationStarted: () => {},
        onPositionsSynchronized: () => {},
        onPendingOrdersSynchronized: () => {},
        onPendingOrdersReplaced: () => {},
        onAccountInformationUpdated: () => {}
      };
      sandbox.stub(listener, 'onSynchronizationStarted').resolves();
      sandbox.stub(listener, 'onPositionsSynchronized').resolves();
      sandbox.stub(listener, 'onPendingOrdersSynchronized').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'synchronizationStarted', accountId: 'accountId', instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1', positionsUpdated: false,
        ordersUpdated: true });
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation, instanceIndex: 1, host: 'ps-mpa-1', synchronizationId: 'synchronizationId' });
      await new _promise2.default(res => setTimeout(res, 100));
      _sinon2.default.assert.calledWith(listener.onSynchronizationStarted, '1:ps-mpa-1', true, false, true);
      _sinon2.default.assert.calledWith(listener.onPositionsSynchronized, '1:ps-mpa-1', 'synchronizationId');
      _sinon2.default.assert.notCalled(listener.onPendingOrdersSynchronized);
      server.emit('synchronization', { type: 'orders', accountId: 'accountId', orders, instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1' });
      await new _promise2.default(res => setTimeout(res, 100));
      _sinon2.default.assert.calledWith(listener.onPendingOrdersSynchronized, '1:ps-mpa-1', 'synchronizationId');
    });

    it('should process synchronization started event without updating orders', async () => {
      let positions = [{
        id: '46214692',
        type: 'POSITION_TYPE_BUY',
        symbol: 'GBPUSD',
        magic: 1000,
        time: '2020-04-15T02:45:06.521Z',
        updateTime: '2020-04-15T02:45:06.521Z',
        openPrice: 1.26101,
        currentPrice: 1.24883,
        currentTickValue: 1,
        volume: 0.07,
        swap: 0,
        profit: -85.25999999999966,
        commission: -0.25,
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        stopLoss: 1.17721,
        unrealizedProfit: -85.25999999999901,
        realizedProfit: -6.536993168992922e-13
      }];

      client._socketInstances[0].synchronizationThrottler = synchronizationThrottler;
      let listener = {
        onSynchronizationStarted: () => {},
        onPositionsSynchronized: () => {},
        onPendingOrdersSynchronized: () => {},
        onPositionsReplaced: () => {},
        onAccountInformationUpdated: () => {}
      };
      sandbox.stub(listener, 'onSynchronizationStarted').resolves();
      sandbox.stub(listener, 'onPositionsSynchronized').resolves();
      sandbox.stub(listener, 'onPendingOrdersSynchronized').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'synchronizationStarted', accountId: 'accountId', instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1', positionsUpdated: true,
        ordersUpdated: false });
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        accountInformation, instanceIndex: 1, host: 'ps-mpa-1', synchronizationId: 'synchronizationId' });
      await new _promise2.default(res => setTimeout(res, 100));
      _sinon2.default.assert.calledWith(listener.onSynchronizationStarted, '1:ps-mpa-1', true, true, false);
      _sinon2.default.assert.notCalled(listener.onPositionsSynchronized);
      _sinon2.default.assert.notCalled(listener.onPendingOrdersSynchronized);
      server.emit('synchronization', { type: 'positions', accountId: 'accountId', positions, instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onPositionsSynchronized, '1:ps-mpa-1', 'synchronizationId');
      _sinon2.default.assert.calledWith(listener.onPendingOrdersSynchronized, '1:ps-mpa-1', 'synchronizationId');
    });

    it('should synchronize account information', async () => {
      let listener = {
        onAccountInformationUpdated: () => {}
      };
      sandbox.stub(listener, 'onAccountInformationUpdated').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'accountInformation', accountId: 'accountId',
        host: 'ps-mpa-1', accountInformation, instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onAccountInformationUpdated, '1:ps-mpa-1', accountInformation);
    });

    it('should synchronize positions', async () => {
      let positions = [{
        id: '46214692',
        type: 'POSITION_TYPE_BUY',
        symbol: 'GBPUSD',
        magic: 1000,
        time: new Date('2020-04-15T02:45:06.521Z'),
        updateTime: new Date('2020-04-15T02:45:06.521Z'),
        openPrice: 1.26101,
        currentPrice: 1.24883,
        currentTickValue: 1,
        volume: 0.07,
        swap: 0,
        profit: -85.25999999999966,
        commission: -0.25,
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        stopLoss: 1.17721,
        unrealizedProfit: -85.25999999999901,
        realizedProfit: -6.536993168992922e-13
      }];
      client._socketInstances[0].synchronizationThrottler = synchronizationThrottler;
      let listener = {
        onPositionsReplaced: () => {},
        onPositionsSynchronized: () => {}
      };
      sandbox.stub(listener, 'onPositionsReplaced').resolves();
      sandbox.stub(listener, 'onPositionsSynchronized').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'positions', accountId: 'accountId', positions, instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onPositionsReplaced, '1:ps-mpa-1', positions);
      _sinon2.default.assert.calledWith(listener.onPositionsSynchronized, '1:ps-mpa-1', 'synchronizationId');
    });

    it('should synchronize orders', async () => {
      let orders = [{
        id: '46871284',
        type: 'ORDER_TYPE_BUY_LIMIT',
        state: 'ORDER_STATE_PLACED',
        symbol: 'AUDNZD',
        magic: 123456,
        platform: 'mt5',
        time: new Date('2020-04-20T08:38:58.270Z'),
        openPrice: 1.03,
        currentPrice: 1.05206,
        volume: 0.01,
        currentVolume: 0.01,
        comment: 'COMMENT2'
      }];
      let listener = {
        onPendingOrdersReplaced: () => {},
        onPendingOrdersSynchronized: () => {}
      };
      client._socketInstances[0].synchronizationThrottler = synchronizationThrottler;
      sandbox.stub(listener, 'onPendingOrdersReplaced').resolves();
      sandbox.stub(listener, 'onPendingOrdersSynchronized').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'orders', accountId: 'accountId', orders, instanceIndex: 1,
        synchronizationId: 'synchronizationId', host: 'ps-mpa-1' });
      await new _promise2.default(res => setTimeout(res, 100));
      _sinon2.default.assert.calledWith(listener.onPendingOrdersReplaced, '1:ps-mpa-1', orders);
      _sinon2.default.assert.calledWith(listener.onPendingOrdersSynchronized, '1:ps-mpa-1', 'synchronizationId');
    });

    it('should synchronize history orders', async () => {
      let historyOrders = [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        currentPrice: 1.261,
        currentVolume: 0,
        doneTime: new Date('2020-04-15T02:45:06.521Z'),
        id: '46214692',
        magic: 1000,
        platform: 'mt5',
        positionId: '46214692',
        state: 'ORDER_STATE_FILLED',
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.260Z'),
        type: 'ORDER_TYPE_BUY',
        volume: 0.07
      }];
      let listener = {
        onHistoryOrderAdded: () => {}
      };
      sandbox.stub(listener, 'onHistoryOrderAdded').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'historyOrders', accountId: 'accountId', historyOrders,
        instanceIndex: 1, host: 'ps-mpa-1' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onHistoryOrderAdded, '1:ps-mpa-1', historyOrders[0]);
    });

    it('should synchronize deals', async () => {
      let deals = [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        commission: -0.25,
        entryType: 'DEAL_ENTRY_IN',
        id: '33230099',
        magic: 1000,
        platform: 'mt5',
        orderId: '46214692',
        positionId: '46214692',
        price: 1.26101,
        profit: 0,
        swap: 0,
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.521Z'),
        type: 'DEAL_TYPE_BUY',
        volume: 0.07
      }];
      let listener = {
        onDealAdded: () => {}
      };
      sandbox.stub(listener, 'onDealAdded').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'deals', accountId: 'accountId', deals, instanceIndex: 1,
        host: 'ps-mpa-1' });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onDealAdded, '1:ps-mpa-1', deals[0]);
    });

    it('should process synchronization updates', async () => {
      let update = {
        accountInformation: {
          broker: 'True ECN Trading Ltd',
          currency: 'USD',
          server: 'ICMarketsSC-Demo',
          balance: 7319.9,
          equity: 7306.649913200001,
          margin: 184.1,
          freeMargin: 7120.22,
          leverage: 100,
          marginLevel: 3967.58283542
        },
        updatedPositions: [{
          id: '46214692',
          type: 'POSITION_TYPE_BUY',
          symbol: 'GBPUSD',
          magic: 1000,
          time: new Date('2020-04-15T02:45:06.521Z'),
          updateTime: new Date('2020-04-15T02:45:06.521Z'),
          openPrice: 1.26101,
          currentPrice: 1.24883,
          currentTickValue: 1,
          volume: 0.07,
          swap: 0,
          profit: -85.25999999999966,
          commission: -0.25,
          clientId: 'TE_GBPUSD_7hyINWqAlE',
          stopLoss: 1.17721,
          unrealizedProfit: -85.25999999999901,
          realizedProfit: -6.536993168992922e-13
        }],
        removedPositionIds: ['1234'],
        updatedOrders: [{
          id: '46871284',
          type: 'ORDER_TYPE_BUY_LIMIT',
          state: 'ORDER_STATE_PLACED',
          symbol: 'AUDNZD',
          magic: 123456,
          platform: 'mt5',
          time: new Date('2020-04-20T08:38:58.270Z'),
          openPrice: 1.03,
          currentPrice: 1.05206,
          volume: 0.01,
          currentVolume: 0.01,
          comment: 'COMMENT2'
        }],
        completedOrderIds: ['2345'],
        historyOrders: [{
          clientId: 'TE_GBPUSD_7hyINWqAlE',
          currentPrice: 1.261,
          currentVolume: 0,
          doneTime: new Date('2020-04-15T02:45:06.521Z'),
          id: '46214692',
          magic: 1000,
          platform: 'mt5',
          positionId: '46214692',
          state: 'ORDER_STATE_FILLED',
          symbol: 'GBPUSD',
          time: new Date('2020-04-15T02:45:06.260Z'),
          type: 'ORDER_TYPE_BUY',
          volume: 0.07
        }],
        deals: [{
          clientId: 'TE_GBPUSD_7hyINWqAlE',
          commission: -0.25,
          entryType: 'DEAL_ENTRY_IN',
          id: '33230099',
          magic: 1000,
          platform: 'mt5',
          orderId: '46214692',
          positionId: '46214692',
          price: 1.26101,
          profit: 0,
          swap: 0,
          symbol: 'GBPUSD',
          time: new Date('2020-04-15T02:45:06.521Z'),
          type: 'DEAL_TYPE_BUY',
          volume: 0.07
        }]
      };
      let listener = {
        onAccountInformationUpdated: () => {},
        onPositionUpdated: () => {},
        onPositionRemoved: () => {},
        onPendingOrderUpdated: () => {},
        onPendingOrderCompleted: () => {},
        onHistoryOrderAdded: () => {},
        onDealAdded: () => {}
      };
      sandbox.stub(listener, 'onAccountInformationUpdated').resolves();
      sandbox.stub(listener, 'onPositionUpdated').resolves();
      sandbox.stub(listener, 'onPositionRemoved').resolves();
      sandbox.stub(listener, 'onPendingOrderUpdated').resolves();
      sandbox.stub(listener, 'onPendingOrderCompleted').resolves();
      sandbox.stub(listener, 'onHistoryOrderAdded').resolves();
      sandbox.stub(listener, 'onDealAdded').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', (0, _assign2.default)({ type: 'update', accountId: 'accountId', instanceIndex: 1,
        host: 'ps-mpa-1' }, update));
      await new _promise2.default(res => setTimeout(res, 100));
      _sinon2.default.assert.calledWith(listener.onAccountInformationUpdated, '1:ps-mpa-1', update.accountInformation);
      _sinon2.default.assert.calledWith(listener.onPositionUpdated, '1:ps-mpa-1', update.updatedPositions[0]);
      _sinon2.default.assert.calledWith(listener.onPositionRemoved, '1:ps-mpa-1', update.removedPositionIds[0]);
      _sinon2.default.assert.calledWith(listener.onPendingOrderUpdated, '1:ps-mpa-1', update.updatedOrders[0]);
      _sinon2.default.assert.calledWith(listener.onPendingOrderCompleted, '1:ps-mpa-1', update.completedOrderIds[0]);
      _sinon2.default.assert.calledWith(listener.onHistoryOrderAdded, '1:ps-mpa-1', update.historyOrders[0]);
      _sinon2.default.assert.calledWith(listener.onDealAdded, '1:ps-mpa-1', update.deals[0]);
    });
  });

  describe('market data synchronization', () => {

    beforeEach(() => {
      sandbox.stub(client._subscriptionManager, 'isSubscriptionActive').returns(true);
    });

    afterEach(() => {
      client.removeAllListeners();
    });

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should retry request on failure', async () => {
      let requestCounter = 0;
      let order = {
        id: '46871284',
        type: 'ORDER_TYPE_BUY_LIMIT',
        state: 'ORDER_STATE_PLACED',
        symbol: 'AUDNZD',
        magic: 123456,
        platform: 'mt5',
        time: new Date('2020-04-20T08:38:58.270Z'),
        openPrice: 1.03,
        currentPrice: 1.05206,
        volume: 0.01,
        currentVolume: 0.01,
        comment: 'COMMENT2'
      };
      server.on('request', data => {
        if (requestCounter > 1 && data.type === 'getOrder' && data.accountId === 'accountId' && data.orderId === '46871284' && data.application === 'RPC') {
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, order });
        }
        requestCounter++;
      });
      let actual = await client.getOrder('accountId', '46871284');
      actual.should.match(order);
    }).timeout(20000);

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should wait specified amount of time on too many requests error', async () => {
      let requestCounter = 0;
      let order = {
        id: '46871284',
        type: 'ORDER_TYPE_BUY_LIMIT',
        state: 'ORDER_STATE_PLACED',
        symbol: 'AUDNZD',
        magic: 123456,
        platform: 'mt5',
        time: new Date('2020-04-20T08:38:58.270Z'),
        openPrice: 1.03,
        currentPrice: 1.05206,
        volume: 0.01,
        currentVolume: 0.01,
        comment: 'COMMENT2'
      };
      server.on('request', data => {
        if (requestCounter > 0 && data.type === 'getOrder' && data.accountId === 'accountId' && data.orderId === '46871284' && data.application === 'RPC') {
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, order });
        } else {
          server.emit('processingError', {
            id: 1, error: 'TooManyRequestsError', requestId: data.requestId,
            message: 'The API allows 10000 requests per 60 minutes to avoid overloading our servers.',
            status_code: 429, metadata: {
              periodInMinutes: 60, maxRequestsForPeriod: 10000,
              recommendedRetryTime: new Date(Date.now() + 1000)
            }
          });
        }
        requestCounter++;
      });
      const startTime = Date.now();
      let actual = await client.getOrder('accountId', '46871284');
      actual.should.match(order);
      (Date.now() - startTime).should.be.approximately(1000, 100);
    }).timeout(10000);

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should return too many requests exception if recommended time is beyond max request time', async () => {
      let requestCounter = 0;
      let order = {
        id: '46871284',
        type: 'ORDER_TYPE_BUY_LIMIT',
        state: 'ORDER_STATE_PLACED',
        symbol: 'AUDNZD',
        magic: 123456,
        platform: 'mt5',
        time: new Date('2020-04-20T08:38:58.270Z'),
        openPrice: 1.03,
        currentPrice: 1.05206,
        volume: 0.01,
        currentVolume: 0.01,
        comment: 'COMMENT2'
      };
      server.on('request', data => {
        if (requestCounter > 0 && data.type === 'getOrder' && data.accountId === 'accountId' && data.orderId === '46871284' && data.application === 'RPC') {
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, order });
        } else {
          server.emit('processingError', {
            id: 1, error: 'TooManyRequestsError', requestId: data.requestId,
            message: 'The API allows 10000 requests per 60 minutes to avoid overloading our servers.',
            status_code: 429, metadata: {
              periodInMinutes: 60, maxRequestsForPeriod: 10000,
              recommendedRetryTime: new Date(Date.now() + 60000)
            }
          });
        }
        requestCounter++;
      });

      try {
        await client.getOrder('accountId', '46871284');
        throw new Error('TooManyRequestsError expected');
      } catch (err) {
        err.name.should.equal('TooManyRequestsError');
      }
    }).timeout(10000);

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should not retry request on validation error', async () => {
      let requestCounter = 0;
      server.on('request', data => {
        if (requestCounter > 0 && data.type === 'subscribeToMarketData' && data.accountId === 'accountId' && data.symbol === 'EURUSD' && data.application === 'application' && data.instanceIndex === 1) {
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        } else {
          server.emit('processingError', {
            id: 1, error: 'ValidationError', message: 'Error message', requestId: data.requestId
          });
        }
        requestCounter++;
      });
      try {
        await client.subscribeToMarketData('accountId', 1, 'EURUSD');
        throw new Error('ValidationError expected');
      } catch (err) {
        err.name.should.equal('ValidationError');
      }
      _sinon2.default.assert.match(requestCounter, 1);
    }).timeout(6000);

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should not retry trade requests on fail', async () => {
      let requestCounter = 0;
      let trade = {
        actionType: 'ORDER_TYPE_SELL',
        symbol: 'AUDNZD',
        volume: 0.07
      };
      server.on('request', data => {
        if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
          if (requestCounter > 0) {
            _sinon2.default.assert.fail();
          }
          requestCounter++;
        }
      });
      try {
        await client.trade(trade);
        throw new Error('TimeoutError expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
    }).timeout(6000);

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should not retry request if connection closed between retries', async () => {
      let requestCounter = 0;
      let response = { type: 'response', accountId: 'accountId' };
      server.on('request', async data => {
        if (data.type === 'unsubscribe' && data.accountId === 'accountId') {
          server.emit('response', (0, _assign2.default)({ requestId: data.requestId }, response));
        }

        if (data.type === 'getOrders' && data.accountId === 'accountId' && data.application === 'RPC') {
          requestCounter++;
          server.emit('processingError', {
            id: 1, error: 'NotSynchronizedError', message: 'Error message',
            requestId: data.requestId
          });
        }
      });
      client.unsubscribe('accountId');
      try {
        await client.getOrders('accountId');
        throw new Error('NotSynchronizedError expected');
      } catch (err) {
        err.name.should.equal('NotSynchronizedError');
      }
      requestCounter.should.equal(1);
      client.socketInstancesByAccounts.should.not.have.property('accountId');
    });

    /**
     * @test {MetaApiWebsocketClient#rpcRequest}
     */
    it('should return timeout error if no server response received', async () => {
      let trade = {
        actionType: 'ORDER_TYPE_SELL',
        symbol: 'AUDNZD',
        volume: 0.07
      };
      server.on('request', data => {});
      try {
        await client.trade(trade);
        throw new Error('TimeoutError extected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
    }).timeout(20000);

    /**
     * @test {MetaApiWebsocketClient#subscribeToMarketData}
     */
    it('should subscribe to market data with MetaTrader terminal', async () => {
      let requestReceived = false;
      server.on('request', data => {
        if (data.type === 'subscribeToMarketData' && data.accountId === 'accountId' && data.symbol === 'EURUSD' && data.application === 'application' && data.instanceIndex === 1 && (0, _stringify2.default)(data.subscriptions) === (0, _stringify2.default)([{ type: 'quotes' }])) {
          requestReceived = true;
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        }
      });
      await client.subscribeToMarketData('accountId', 1, 'EURUSD', [{ type: 'quotes' }]);
      requestReceived.should.be.true();
    });

    /**
     * @test {MetaApiWebsocketClient#refreshMarketDataSubscriptions}
     */
    it('should refresh market data subscriptions', async () => {
      let requestReceived = false;
      server.on('request', data => {
        if (data.type === 'refreshMarketDataSubscriptions' && data.accountId === 'accountId' && data.application === 'application' && data.instanceIndex === 1 && (0, _stringify2.default)(data.subscriptions) === (0, _stringify2.default)([{ symbol: 'EURUSD' }])) {
          requestReceived = true;
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        }
      });
      await client.refreshMarketDataSubscriptions('accountId', 1, [{ symbol: 'EURUSD' }]);
      requestReceived.should.be.true();
    });

    /**
     * @test {MetaApiWebsocketClient#unsubscribeFromMarketData}
     */
    it('should unsubscribe from market data with MetaTrader terminal', async () => {
      let requestReceived = false;
      server.on('request', data => {
        if (data.type === 'unsubscribeFromMarketData' && data.accountId === 'accountId' && data.symbol === 'EURUSD' && data.application === 'application' && data.instanceIndex === 1 && (0, _stringify2.default)(data.subscriptions) === (0, _stringify2.default)([{ type: 'quotes' }])) {
          requestReceived = true;
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        }
      });
      await client.unsubscribeFromMarketData('accountId', 1, 'EURUSD', [{ type: 'quotes' }]);
      requestReceived.should.be.true();
    });

    it('should synchronize symbol specifications', async () => {
      let specifications = [{
        symbol: 'EURUSD',
        tickSize: 0.00001,
        minVolume: 0.01,
        maxVolume: 200,
        volumeStep: 0.01
      }];
      let listener = {
        onSymbolSpecificationsUpdated: () => {},
        onSymbolSpecificationUpdated: () => {},
        onSymbolSpecificationRemoved: () => {}
      };
      sandbox.stub(listener, 'onSymbolSpecificationsUpdated').resolves();
      sandbox.stub(listener, 'onSymbolSpecificationUpdated').resolves();
      sandbox.stub(listener, 'onSymbolSpecificationRemoved').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'specifications', accountId: 'accountId', specifications, instanceIndex: 1, host: 'ps-mpa-1',
        removedSymbols: ['AUDNZD'] });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onSymbolSpecificationsUpdated, '1:ps-mpa-1', specifications, ['AUDNZD']);
      _sinon2.default.assert.calledWith(listener.onSymbolSpecificationUpdated, '1:ps-mpa-1', specifications[0]);
      _sinon2.default.assert.calledWith(listener.onSymbolSpecificationRemoved, '1:ps-mpa-1', 'AUDNZD');
    });

    it('should synchronize symbol prices', async () => {
      let prices = [{
        symbol: 'AUDNZD',
        bid: 1.05916,
        ask: 1.05927,
        profitTickValue: 0.602,
        lossTickValue: 0.60203
      }];
      let ticks = [{
        symbol: 'AUDNZD',
        time: new Date('2020-04-07T03:45:00.000Z'),
        brokerTime: '2020-04-07 06:45:00.000',
        bid: 1.05297,
        ask: 1.05309,
        last: 0.5298,
        volume: 0.13,
        side: 'buy'
      }];
      let candles = [{
        symbol: 'AUDNZD',
        timeframe: '15m',
        time: new Date('2020-04-07T03:45:00.000Z'),
        brokerTime: '2020-04-07 06:45:00.000',
        open: 1.03297,
        high: 1.06309,
        low: 1.02705,
        close: 1.043,
        tickVolume: 1435,
        spread: 17,
        volume: 345
      }];
      let books = [{
        symbol: 'AUDNZD',
        time: new Date('2020-04-07T03:45:00.000Z'),
        brokerTime: '2020-04-07 06:45:00.000',
        book: [{
          type: 'BOOK_TYPE_SELL',
          price: 1.05309,
          volume: 5.67
        }, {
          type: 'BOOK_TYPE_BUY',
          price: 1.05297,
          volume: 3.45
        }]
      }];
      let listener = {
        onSymbolPriceUpdated: () => {},
        onSymbolPricesUpdated: () => {},
        onCandlesUpdated: () => {},
        onTicksUpdated: () => {},
        onBooksUpdated: () => {}
      };
      sandbox.stub(listener, 'onSymbolPriceUpdated').resolves();
      sandbox.stub(listener, 'onSymbolPricesUpdated').resolves();
      sandbox.stub(listener, 'onCandlesUpdated').resolves();
      sandbox.stub(listener, 'onTicksUpdated').resolves();
      sandbox.stub(listener, 'onBooksUpdated').resolves();
      client.addSynchronizationListener('accountId', listener);
      server.emit('synchronization', { type: 'prices', accountId: 'accountId', host: 'ps-mpa-1', prices,
        ticks, candles, books, equity: 100, margin: 200, freeMargin: 400, marginLevel: 40000, instanceIndex: 1 });
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.calledWith(listener.onSymbolPricesUpdated, '1:ps-mpa-1', prices, 100, 200, 400, 40000);
      _sinon2.default.assert.calledWith(listener.onCandlesUpdated, '1:ps-mpa-1', candles, 100, 200, 400, 40000);
      _sinon2.default.assert.calledWith(listener.onTicksUpdated, '1:ps-mpa-1', ticks, 100, 200, 400, 40000);
      _sinon2.default.assert.calledWith(listener.onBooksUpdated, '1:ps-mpa-1', books, 100, 200, 400, 40000);
      _sinon2.default.assert.calledWith(listener.onSymbolPriceUpdated, '1:ps-mpa-1', prices[0]);
    });
  });

  describe('wait for server-side terminal state synchronization', () => {

    afterEach(() => {
      client.removeAllListeners();
    });

    /**
     * @test {MetaApiWebsocketClient#waitSynchronized}
     */
    it('should wait for server-side terminal state synchronization', async () => {
      let requestReceived = false;
      server.on('request', data => {
        if (data.type === 'waitSynchronized' && data.accountId === 'accountId' && data.applicationPattern === 'app.*' && data.timeoutInSeconds === 10 && data.application === 'application' && data.instanceIndex === 1) {
          requestReceived = true;
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
        }
      });
      await client.waitSynchronized('accountId', 1, 'app.*', 10);
      requestReceived.should.be.true();
    });
  });

  describe('latency monitoring', () => {

    beforeEach(() => {
      sandbox.stub(client._subscriptionManager, 'isSubscriptionActive').returns(true);
    });

    /**
     * @test {LatencyListener#onResponse}
     */
    it('should invoke latency listener on response', async () => {
      let accountId;
      let requestType;
      let actualTimestamps;
      let listener = {
        onResponse: (aid, type, ts) => {
          accountId = aid;
          requestType = type;
          actualTimestamps = ts;
        }
      };
      client.addLatencyListener(listener);
      let price = {};
      let timestamps;
      server.on('request', data => {
        if (data.type === 'getSymbolPrice' && data.accountId === 'accountId' && data.symbol === 'AUDNZD' && data.application === 'RPC' && data.timestamps.clientProcessingStarted) {
          timestamps = (0, _assign2.default)(data.timestamps, { serverProcessingStarted: new Date(),
            serverProcessingFinished: new Date() });
          timestamps.clientProcessingStarted = new Date(timestamps.clientProcessingStarted);
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, price,
            timestamps });
        }
      });
      await client.getSymbolPrice('accountId', 'AUDNZD');
      await new _promise2.default(res => setTimeout(res, 100));
      accountId.should.equal('accountId');
      requestType.should.equal('getSymbolPrice');
      actualTimestamps.should.match(timestamps);
      _should2.default.exist(actualTimestamps.clientProcessingStarted);
      _should2.default.exist(actualTimestamps.clientProcessingFinished);
      _should2.default.exist(actualTimestamps.serverProcessingStarted);
      _should2.default.exist(actualTimestamps.serverProcessingFinished);
    });

    /**
     * @test {LatencyListener#onSymbolPrice}
     */
    it('should measure price streaming latencies', async () => {
      let prices = [{
        symbol: 'AUDNZD',
        timestamps: {
          eventGenerated: new Date(),
          serverProcessingStarted: new Date(),
          serverProcessingFinished: new Date()
        }
      }];
      let accountId;
      let symbol;
      let actualTimestamps;
      let listener = {
        onSymbolPrice: (aid, sym, ts) => {
          accountId = aid;
          symbol = sym;
          actualTimestamps = ts;
        }
      };
      client.addLatencyListener(listener);
      server.emit('synchronization', { type: 'prices', accountId: 'accountId', prices, equity: 100, margin: 200,
        freeMargin: 400, marginLevel: 40000 });
      await new _promise2.default(res => setTimeout(res, 50));
      accountId.should.equal('accountId');
      symbol.should.equal('AUDNZD');
      actualTimestamps.should.match(prices[0].timestamps);
      _should2.default.exist(actualTimestamps.clientProcessingFinished);
    });

    /**
     * @test {LatencyListener#onUpdate}
     */
    it('should measure update latencies', async () => {
      let update = {
        timestamps: {
          eventGenerated: new Date(),
          serverProcessingStarted: new Date(),
          serverProcessingFinished: new Date()
        }
      };
      let accountId;
      let actualTimestamps;
      let listener = {
        onUpdate: (aid, ts) => {
          accountId = aid;
          actualTimestamps = ts;
        }
      };
      client.addLatencyListener(listener);
      server.emit('synchronization', (0, _assign2.default)({ type: 'update', accountId: 'accountId' }, update));
      await new _promise2.default(res => setTimeout(res, 50));
      accountId.should.equal('accountId');
      actualTimestamps.should.match(update.timestamps);
      _should2.default.exist(actualTimestamps.clientProcessingFinished);
    });

    /**
     * @test {LatencyListener#onTrade}
     */
    it('should process trade latency', async () => {
      let trade = {};
      let response = {
        numericCode: 10009,
        stringCode: 'TRADE_RETCODE_DONE',
        message: 'Request completed',
        orderId: '46870472'
      };
      let timestamps = {
        clientExecutionStarted: new Date(),
        serverExecutionStarted: new Date(),
        serverExecutionFinished: new Date(),
        tradeExecuted: new Date()
      };
      let accountId;
      let actualTimestamps;
      let listener = {
        onTrade: (aid, ts) => {
          accountId = aid;
          actualTimestamps = ts;
        }
      };
      client.addLatencyListener(listener);
      server.on('request', data => {
        data.trade.should.match(trade);
        if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
          server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, response,
            timestamps });
        }
      });
      await client.trade('accountId', trade);
      accountId.should.equal('accountId');
      actualTimestamps.should.match(timestamps);
      _should2.default.exist(actualTimestamps.clientProcessingFinished);
    });
  });

  it('should reconnect to server on disconnect', async () => {
    const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
    const trade = {
      actionType: 'ORDER_TYPE_SELL',
      symbol: 'AUDNZD',
      volume: 0.07
    };
    const response = {
      numericCode: 10009,
      stringCode: 'TRADE_RETCODE_DONE',
      message: 'Request completed',
      orderId: '46870472'
    };
    let listener = {
      onReconnected: () => {}
    };
    sandbox.stub(listener, 'onReconnected').resolves();
    sandbox.stub(client._packetOrderer, 'onReconnected').resolves();
    sandbox.stub(client._subscriptionManager, 'onReconnected').resolves();
    client.addReconnectListener(listener, 'accountId');
    let requestCounter = 0;
    server.on('request', async data => {
      if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
        requestCounter++;
        await server.emit('response', { type: 'response', accountId: data.accountId,
          requestId: data.requestId, response });
      }
      await server.disconnect();
    });

    client.trade('accountId', trade);
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(1500);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledOnce(listener.onReconnected);
    _sinon2.default.assert.calledWith(client._subscriptionManager.onReconnected, 0, ['accountId']);
    _sinon2.default.assert.calledWith(client._packetOrderer.onReconnected, ['accountId']);

    server.on('request', async data => {
      if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
        requestCounter++;
        await server.emit('response', { type: 'response', accountId: data.accountId,
          requestId: data.requestId, response });
      }
      await server.disconnect();
    });

    client.trade('accountId', trade);
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(1500);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.match(requestCounter, 2);
    clock.restore();
  });

  /**
   * @test {MetaApiWebsocketClient#rpcRequest}
   */
  it('should remove reconnect listener', async () => {
    const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });

    let trade = {
      actionType: 'ORDER_TYPE_SELL',
      symbol: 'AUDNZD',
      volume: 0.07
    };
    let response = {
      numericCode: 10009,
      stringCode: 'TRADE_RETCODE_DONE',
      message: 'Request completed',
      orderId: '46870472'
    };
    const listener = { onReconnected: async () => {} };
    sandbox.stub(listener, 'onReconnected').resolves();
    client.addReconnectListener(listener, 'accountId');
    sandbox.stub(client._subscriptionManager, 'onReconnected');
    let requestCounter = 0;
    server.on('request', data => {
      data.trade.should.match(trade);
      requestCounter++;
      if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, response });
      }
      server.disconnect();
    });

    await client.trade('accountId', trade);
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(1100);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledOnce(listener.onReconnected);
    client.removeReconnectListener(listener);

    server.on('request', data => {
      data.trade.should.match(trade);
      requestCounter++;
      if (data.type === 'trade' && data.accountId === 'accountId' && data.application === 'application') {
        server.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId, response });
      }
      server.disconnect();
    });

    await client.trade('accountId', trade);
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(1100);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledOnce(listener.onReconnected);
    _sinon2.default.assert.match(requestCounter, 2);
    clock.restore();
  });

  /**
   * @test {MetaApiWebsocketClient#queuePacket}
   */
  it('should process packets in order', async () => {
    const clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
    let ordersCallTime = 0;
    let positionsCallTime = 0;
    let disconnectedCallTime = 0;
    let pricesCallTime = 0;
    let listener = {
      onConnected: () => {},
      onDisconnected: async () => {
        await new _promise2.default(res => setTimeout(res, 5000));
        disconnectedCallTime = Date.now();
      },
      onPendingOrdersReplaced: async () => {
        await new _promise2.default(res => setTimeout(res, 10000));
        ordersCallTime = Date.now();
      },
      onPendingOrdersSynchronized: () => {},
      onPositionsReplaced: async () => {
        await new _promise2.default(res => setTimeout(res, 1000));
        positionsCallTime = Date.now();
      },
      onPositionsSynchronized: () => {},
      onSymbolPriceUpdated: () => {},
      onSymbolPricesUpdated: async () => {
        await new _promise2.default(res => setTimeout(res, 1000));
        pricesCallTime = Date.now();
      }
    };
    let resolve;
    let promise = new _promise2.default(res => resolve = res);
    client.close();
    io.close(() => resolve());
    await promise;
    io = new _socket2.default(6785, { path: '/ws', pingTimeout: 1000000 });
    sandbox.stub(httpClient, 'request').resolves({ url: 'http://localhost:6785' });
    client = new _metaApiWebsocket2.default(httpClient, 'token', { application: 'application',
      domain: 'project-stock.agiliumlabs.cloud', requestTimeout: 1.5, useSharedClientApi: false,
      retryOpts: { retries: 3, minDelayInSeconds: 0.1, maxDelayInSeconds: 0.5 },
      eventProcessing: { sequentialProcessing: true } });
    sandbox.stub(client._subscriptionManager, 'isSubscriptionActive').returns(true);
    io.on('connect', socket => {
      server = socket;
      if (socket.request._query['auth-token'] !== 'token') {
        socket.emit({ error: 'UnauthorizedError', message: 'Authorization token invalid' });
        socket.close();
      }
      server.on('request', data => {
        if (data.type === 'getPositions' && data.accountId === 'accountId' && data.application === 'RPC') {
          server.emit('response', { type: 'response', accountId: data.accountId,
            requestId: data.requestId, positions: [] });
        }
      });
    });
    await client.getPositions('accountId');
    client.addSynchronizationListener('accountId', listener);
    sandbox.stub(client._packetOrderer, 'restoreOrder').callsFake(arg => {
      return [arg];
    });
    server.emit('synchronization', { type: 'authenticated', accountId: 'accountId', host: 'ps-mpa-1',
      instanceIndex: 1, replicas: 2, sequenceNumber: 1 });
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(59000);
    server.emit('synchronization', { type: 'orders', accountId: 'accountId', orders: [], instanceIndex: 1,
      host: 'ps-mpa-1', sequenceNumber: 2 });
    server.emit('synchronization', { type: 'prices', accountId: 'accountId', prices: [{ symbol: 'EURUSD' }],
      instanceIndex: 1, host: 'ps-mpa-1', equity: 100, margin: 200, freeMargin: 400, marginLevel: 40000 });
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(3000);
    server.emit('synchronization', { type: 'positions', accountId: 'accountId', positions: [], instanceIndex: 1,
      host: 'ps-mpa-1', sequenceNumber: 3 });
    await new _promise2.default(res => setTimeout(res, 50));
    await clock.tickAsync(20000);
    await new _promise2.default(res => setTimeout(res, 50));
    pricesCallTime.should.not.eql(0);
    ordersCallTime.should.be.above(pricesCallTime);
    disconnectedCallTime.should.be.above(ordersCallTime);
    positionsCallTime.should.be.above(disconnectedCallTime);
    clock.restore();
  });

  /**
   * @test {MetaApiWebsocketClient#queuePacket}
   */
  it('should not process old synchronization packet without gaps in sequence numbers', async () => {
    let listener = {
      onSynchronizationStarted: _sinon2.default.fake(),
      onPendingOrdersReplaced: _sinon2.default.fake(),
      onPendingOrdersSynchronized: () => {}
    };
    client.addSynchronizationListener('accountId', listener);
    sandbox.stub(client._subscriptionManager, 'isSubscriptionActive').returns(true);
    sandbox.stub(client._packetOrderer, 'restoreOrder').callsFake(arg => [arg]);

    sandbox.stub(client._socketInstances[0].synchronizationThrottler, 'activeSynchronizationIds').get(() => ['ABC']);
    server.emit('synchronization', { type: 'synchronizationStarted', accountId: 'accountId',
      sequenceNumber: 1, sequenceTimestamp: 1603124267178, synchronizationId: 'ABC' });
    server.emit('synchronization', { type: 'orders', accountId: 'accountId', orders: [],
      sequenceNumber: 2, sequenceTimestamp: 1603124267181, synchronizationId: 'ABC' });
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledOnce(listener.onSynchronizationStarted);
    _sinon2.default.assert.calledOnce(listener.onPendingOrdersReplaced);

    sandbox.stub(client._socketInstances[0].synchronizationThrottler, 'activeSynchronizationIds').get(() => ['DEF']);
    server.emit('synchronization', { type: 'synchronizationStarted', accountId: 'accountId',
      sequenceNumber: 3, sequenceTimestamp: 1603124267190, synchronizationId: 'DEF' });
    server.emit('synchronization', { type: 'orders', accountId: 'accountId', orders: [],
      sequenceNumber: 4, sequenceTimestamp: 1603124267192, synchronizationId: 'ABC' });
    server.emit('synchronization', { type: 'orders', accountId: 'accountId', orders: [],
      sequenceNumber: 5, sequenceTimestamp: 1603124267195, synchronizationId: 'DEF' });
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledTwice(listener.onSynchronizationStarted);
    _sinon2.default.assert.calledTwice(listener.onPendingOrdersReplaced);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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