'use strict';

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _historicalMarketData = require('./historicalMarketData.client');

var _historicalMarketData2 = _interopRequireDefault(_historicalMarketData);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const marketDataClientApiUrl = 'https://mt-market-data-client-api-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {HistoricalMarketDataClient}
 */
describe('HistoricalMarketDataClient', () => {

  let client;
  const token = 'header.payload.sign';
  let httpClient = new _httpClient2.default();
  let sandbox;
  let requestStub;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    client = new _historicalMarketData2.default(httpClient, token);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {HistoricalMarketDataClient#getHistoricalCandles}
   */
  it('should download historical candles from API', async () => {
    let expected = [{
      symbol: 'AUDNZD',
      timeframe: '15m',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      open: 1.03297,
      high: 1.06309,
      low: 1.02705,
      close: 1.043,
      tickVolume: 1435,
      spread: 17,
      volume: 345
    }];
    requestStub.resolves(expected);
    let candles = await client.getHistoricalCandles('accountId', 'AUDNZD', '15m', new Date('2020-04-07T03:45:00.000Z'), 1);
    candles.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${marketDataClientApiUrl}/users/current/accounts/accountId/historical-market-data/symbols/AUDNZD/` + 'timeframes/15m/candles',
      method: 'GET',
      qs: {
        startTime: new Date('2020-04-07T03:45:00.000Z'),
        limit: 1
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {HistoricalMarketDataClient#getHistoricalCandles}
   */
  it('should download historical candles from API for symbol with special characters', async () => {
    let expected = [{
      symbol: 'GBPJPY#',
      timeframe: '15m',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      open: 1.03297,
      high: 1.06309,
      low: 1.02705,
      close: 1.043,
      tickVolume: 1435,
      spread: 17,
      volume: 345
    }];
    requestStub.resolves(expected);
    let candles = await client.getHistoricalCandles('accountId', 'GBPJPY#', '15m', new Date('2020-04-07T03:45:00.000Z'), 1);
    candles.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${marketDataClientApiUrl}/users/current/accounts/accountId/historical-market-data/symbols/GBPJPY%23/` + 'timeframes/15m/candles',
      method: 'GET',
      qs: {
        startTime: new Date('2020-04-07T03:45:00.000Z'),
        limit: 1
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {HistoricalMarketDataClient#getHistoricalTicks}
   */
  it('should download historical ticks from API', async () => {
    let expected = [{
      symbol: 'AUDNZD',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      bid: 1.05297,
      ask: 1.05309,
      last: 0.5298,
      volume: 0.13,
      side: 'buy'
    }];
    requestStub.resolves(expected);
    let ticks = await client.getHistoricalTicks('accountId', 'AUDNZD', new Date('2020-04-07T03:45:00.000Z'), 0, 1);
    ticks.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${marketDataClientApiUrl}/users/current/accounts/accountId/historical-market-data/symbols/AUDNZD/ticks`,
      method: 'GET',
      qs: {
        startTime: new Date('2020-04-07T03:45:00.000Z'),
        offset: 0,
        limit: 1
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {HistoricalMarketDataClient#getHistoricalTicks}
   */
  it('should download historical ticks from API for symbol with special characters', async () => {
    let expected = [{
      symbol: 'GBPJPY#',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      bid: 1.05297,
      ask: 1.05309,
      last: 0.5298,
      volume: 0.13,
      side: 'buy'
    }];
    requestStub.resolves(expected);
    let ticks = await client.getHistoricalTicks('accountId', 'GBPJPY#', new Date('2020-04-07T03:45:00.000Z'), 0, 1);
    ticks.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${marketDataClientApiUrl}/users/current/accounts/accountId/historical-market-data/symbols/GBPJPY%23/ticks`,
      method: 'GET',
      qs: {
        startTime: new Date('2020-04-07T03:45:00.000Z'),
        offset: 0,
        limit: 1
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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