'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud expert advisor API client (see https://metaapi.cloud/docs/provisioning/)
 */
class ExpertAdvisorClient extends _metaApi2.default {

  /**
   * Expert advisor model
   * @typedef {Object} ExpertAdvisorDto
   * @property {String} expertId expert advisor id
   * @property {String} period expert advisor period
   * @property {String} symbol expert advisor symbol
   * @property {Boolean} fileUploaded true if expert file was uploaded
   */

  /**
   * Retrieves expert advisors by account id (see
   * https://metaapi.cloud/docs/provisioning/api/expertAdvisor/readExpertAdvisors/)
   * Method is accessible only with API access token
   * @param {String} accountId Metatrader account id
   * @returns {Promise<ExpertAdvisorDto[]>} promise resolving with expert advisors found
   */
  getExpertAdvisors(accountId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getExpertAdvisors');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${accountId}/expert-advisors`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Retrieves an expert advisor by id (see
   * https://metaapi.cloud/docs/provisioning/api/expertAdvisor/readExpertAdvisor/).
   * Thrown an error if expert is not found. Method is accessible only with API access token
   * @param {String} accountId Metatrader account id
   * @param {String} expertId expert advisor id
   * @returns {Promise<ExpertAdvisorDto>} promise resolving with expert advisor found
   */
  getExpertAdvisor(accountId, expertId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getExpertAdvisors');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${accountId}/expert-advisors/${expertId}`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Updated expert advisor data
   * @typedef {Object} NewExpertAdvisorDto
   * @property {String} period expert advisor symbol.
   * For MetaTrader 4 allowed periods are 1m, 5m, 15m, 30m, 1h, 4h, 1d, 1w, 1mn
   * For MetaTrader 5 allowed periods are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h,
   * 1d, 1w, 1mn
   * @property {String} symbol expert advisor period
   * @property {String} preset base64-encoded preset file
   */

  /**
   * Updates or creates expert advisor data (see
   * https://metaapi.cloud/docs/provisioning/api/expertAdvisor/updateExpertAdvisor/).
   * Method is accessible only with API access token
   * @param {String} accountId Metatrader account id
   * @param {String} expertId expert id
   * @param {NewExpertAdvisorDto} expert new expert advisor data
   * @returns {Promise} promise resolving when expert advisor is updated
   */
  updateExpertAdvisor(accountId, expertId, expert) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getExpertAdvisors');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${accountId}/expert-advisors/${expertId}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: expert
    };
    return this._httpClient.request(opts);
  }

  /**
   * Uploads an expert advisor file (see https://metaapi.cloud/docs/provisioning/api/expertAdvisor/uploadEAFile/)
   * Method is accessible only with API access token
   * @param {String} accountId Metatrader account id
   * @param {String} expertId expert id
   * @param {String|Buffer} file path to a file to upload or buffer containing file contents
   * @returns {Promise} promise resolving when file upload is completed
   */
  uploadExpertAdvisorFile(accountId, expertId, file) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('uploadProvisioningProfileFile');
    }
    if (typeof file === 'string') {
      file = _fs2.default.createReadStream(file);
    }
    const opts = {
      method: 'PUT',
      url: `${this._host}/users/current/accounts/${accountId}/expert-advisors/${expertId}/file`,
      formData: {
        file
      },
      json: true,
      headers: {
        'auth-token': this._token
      }
    };
    return this._httpClient.request(opts);
  }

  /**
   * Deletes an expert advisor (see https://metaapi.cloud/docs/provisioning/api/expertAdvisor/deleteExpertAdvisor/)
   * Method is accessible only with API access token
   * @param {String} accountId Metatrader account id
   * @param {String} expertId expert id
   * @returns {Promise} promise resolving when expert advisor is deleted
   */
  deleteExpertAdvisor(accountId, expertId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getExpertAdvisors');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${accountId}/expert-advisors/${expertId}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

}
exports.default = ExpertAdvisorClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uLy4uL2xpYi9jbGllbnRzL21ldGFBcGkvZXhwZXJ0QWR2aXNvci5jbGllbnQuZXM2Il0sIm5hbWVzIjpbIkV4cGVydEFkdmlzb3JDbGllbnQiLCJNZXRhQXBpQ2xpZW50IiwiZ2V0RXhwZXJ0QWR2aXNvcnMiLCJhY2NvdW50SWQiLCJfaXNOb3RKd3RUb2tlbiIsIl9oYW5kbGVOb0FjY2Vzc0Vycm9yIiwib3B0cyIsInVybCIsIl9ob3N0IiwibWV0aG9kIiwiaGVhZGVycyIsIl90b2tlbiIsImpzb24iLCJfaHR0cENsaWVudCIsInJlcXVlc3QiLCJnZXRFeHBlcnRBZHZpc29yIiwiZXhwZXJ0SWQiLCJ1cGRhdGVFeHBlcnRBZHZpc29yIiwiZXhwZXJ0IiwiYm9keSIsInVwbG9hZEV4cGVydEFkdmlzb3JGaWxlIiwiZmlsZSIsImZzIiwiY3JlYXRlUmVhZFN0cmVhbSIsImZvcm1EYXRhIiwiZGVsZXRlRXhwZXJ0QWR2aXNvciJdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztBQUVBOzs7O0FBQ0E7Ozs7OztBQUVBOzs7QUFHZSxNQUFNQSxtQkFBTixTQUFrQ0MsaUJBQWxDLENBQWdEOztBQUU3RDs7Ozs7Ozs7O0FBU0E7Ozs7Ozs7QUFPQUMsb0JBQWtCQyxTQUFsQixFQUE2QjtBQUMzQixRQUFJLEtBQUtDLGNBQUwsRUFBSixFQUEyQjtBQUN6QixhQUFPLEtBQUtDLG9CQUFMLENBQTBCLG1CQUExQixDQUFQO0FBQ0Q7QUFDRCxVQUFNQyxPQUFPO0FBQ1hDLFdBQU0sR0FBRSxLQUFLQyxLQUFNLDJCQUEwQkwsU0FBVSxrQkFENUM7QUFFWE0sY0FBUSxLQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWEMsWUFBTTtBQU5LLEtBQWI7QUFRQSxXQUFPLEtBQUtDLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUixJQUF6QixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7O0FBUUFTLG1CQUFpQlosU0FBakIsRUFBNEJhLFFBQTVCLEVBQXNDO0FBQ3BDLFFBQUksS0FBS1osY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIsbUJBQTFCLENBQVA7QUFDRDtBQUNELFVBQU1DLE9BQU87QUFDWEMsV0FBTSxHQUFFLEtBQUtDLEtBQU0sMkJBQTBCTCxTQUFVLG9CQUFtQmEsUUFBUyxFQUR4RTtBQUVYUCxjQUFRLEtBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YQyxZQUFNO0FBTkssS0FBYjtBQVFBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJSLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7Ozs7QUFXQTs7Ozs7Ozs7O0FBU0FXLHNCQUFvQmQsU0FBcEIsRUFBK0JhLFFBQS9CLEVBQXlDRSxNQUF6QyxFQUFpRDtBQUMvQyxRQUFJLEtBQUtkLGNBQUwsRUFBSixFQUEyQjtBQUN6QixhQUFPLEtBQUtDLG9CQUFMLENBQTBCLG1CQUExQixDQUFQO0FBQ0Q7QUFDRCxVQUFNQyxPQUFPO0FBQ1hDLFdBQU0sR0FBRSxLQUFLQyxLQUFNLDJCQUEwQkwsU0FBVSxvQkFBbUJhLFFBQVMsRUFEeEU7QUFFWFAsY0FBUSxLQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWEMsWUFBTSxJQU5LO0FBT1hPLFlBQU1EO0FBUEssS0FBYjtBQVNBLFdBQU8sS0FBS0wsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJSLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7QUFRQWMsMEJBQXdCakIsU0FBeEIsRUFBbUNhLFFBQW5DLEVBQTZDSyxJQUE3QyxFQUFtRDtBQUNqRCxRQUFJLEtBQUtqQixjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQiwrQkFBMUIsQ0FBUDtBQUNEO0FBQ0QsUUFBSSxPQUFPZ0IsSUFBUCxLQUFnQixRQUFwQixFQUE4QjtBQUM1QkEsYUFBT0MsYUFBR0MsZ0JBQUgsQ0FBb0JGLElBQXBCLENBQVA7QUFDRDtBQUNELFVBQU1mLE9BQU87QUFDWEcsY0FBUSxLQURHO0FBRVhGLFdBQU0sR0FBRSxLQUFLQyxLQUFNLDJCQUEwQkwsU0FBVSxvQkFBbUJhLFFBQVMsT0FGeEU7QUFHWFEsZ0JBQVU7QUFDUkg7QUFEUSxPQUhDO0FBTVhULFlBQU0sSUFOSztBQU9YRixlQUFTO0FBQ1Asc0JBQWMsS0FBS0M7QUFEWjtBQVBFLEtBQWI7QUFXQSxXQUFPLEtBQUtFLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUixJQUF6QixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7QUFPQW1CLHNCQUFvQnRCLFNBQXBCLEVBQStCYSxRQUEvQixFQUF5QztBQUN2QyxRQUFJLEtBQUtaLGNBQUwsRUFBSixFQUEyQjtBQUN6QixhQUFPLEtBQUtDLG9CQUFMLENBQTBCLG1CQUExQixDQUFQO0FBQ0Q7QUFDRCxVQUFNQyxPQUFPO0FBQ1hDLFdBQU0sR0FBRSxLQUFLQyxLQUFNLDJCQUEwQkwsU0FBVSxvQkFBbUJhLFFBQVMsRUFEeEU7QUFFWFAsY0FBUSxRQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWEMsWUFBTTtBQU5LLEtBQWI7QUFRQSxXQUFPLEtBQUtDLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUixJQUF6QixDQUFQO0FBQ0Q7O0FBN0k0RDtrQkFBMUNOLG1CIiwiZmlsZSI6ImV4cGVydEFkdmlzb3IuY2xpZW50LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5pbXBvcnQgTWV0YUFwaUNsaWVudCBmcm9tICcuLi9tZXRhQXBpLmNsaWVudCc7XG5pbXBvcnQgZnMgZnJvbSAnZnMnO1xuXG4vKipcbiAqIG1ldGFhcGkuY2xvdWQgZXhwZXJ0IGFkdmlzb3IgQVBJIGNsaWVudCAoc2VlIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL3Byb3Zpc2lvbmluZy8pXG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIEV4cGVydEFkdmlzb3JDbGllbnQgZXh0ZW5kcyBNZXRhQXBpQ2xpZW50IHtcblxuICAvKipcbiAgICogRXhwZXJ0IGFkdmlzb3IgbW9kZWxcbiAgICogQHR5cGVkZWYge09iamVjdH0gRXhwZXJ0QWR2aXNvckR0b1xuICAgKiBAcHJvcGVydHkge1N0cmluZ30gZXhwZXJ0SWQgZXhwZXJ0IGFkdmlzb3IgaWRcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IHBlcmlvZCBleHBlcnQgYWR2aXNvciBwZXJpb2RcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IHN5bWJvbCBleHBlcnQgYWR2aXNvciBzeW1ib2xcbiAgICogQHByb3BlcnR5IHtCb29sZWFufSBmaWxlVXBsb2FkZWQgdHJ1ZSBpZiBleHBlcnQgZmlsZSB3YXMgdXBsb2FkZWRcbiAgICovXG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBleHBlcnQgYWR2aXNvcnMgYnkgYWNjb3VudCBpZCAoc2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL3Byb3Zpc2lvbmluZy9hcGkvZXhwZXJ0QWR2aXNvci9yZWFkRXhwZXJ0QWR2aXNvcnMvKVxuICAgKiBNZXRob2QgaXMgYWNjZXNzaWJsZSBvbmx5IHdpdGggQVBJIGFjY2VzcyB0b2tlblxuICAgKiBAcGFyYW0ge1N0cmluZ30gYWNjb3VudElkIE1ldGF0cmFkZXIgYWNjb3VudCBpZFxuICAgKiBAcmV0dXJucyB7UHJvbWlzZTxFeHBlcnRBZHZpc29yRHRvW10+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIGV4cGVydCBhZHZpc29ycyBmb3VuZFxuICAgKi9cbiAgZ2V0RXhwZXJ0QWR2aXNvcnMoYWNjb3VudElkKSB7XG4gICAgaWYgKHRoaXMuX2lzTm90Snd0VG9rZW4oKSkge1xuICAgICAgcmV0dXJuIHRoaXMuX2hhbmRsZU5vQWNjZXNzRXJyb3IoJ2dldEV4cGVydEFkdmlzb3JzJyk7XG4gICAgfVxuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvYWNjb3VudHMvJHthY2NvdW50SWR9L2V4cGVydC1hZHZpc29yc2AsXG4gICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYW4gZXhwZXJ0IGFkdmlzb3IgYnkgaWQgKHNlZVxuICAgKiBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9wcm92aXNpb25pbmcvYXBpL2V4cGVydEFkdmlzb3IvcmVhZEV4cGVydEFkdmlzb3IvKS5cbiAgICogVGhyb3duIGFuIGVycm9yIGlmIGV4cGVydCBpcyBub3QgZm91bmQuIE1ldGhvZCBpcyBhY2Nlc3NpYmxlIG9ubHkgd2l0aCBBUEkgYWNjZXNzIHRva2VuXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBhY2NvdW50SWQgTWV0YXRyYWRlciBhY2NvdW50IGlkXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBleHBlcnRJZCBleHBlcnQgYWR2aXNvciBpZFxuICAgKiBAcmV0dXJucyB7UHJvbWlzZTxFeHBlcnRBZHZpc29yRHRvPn0gcHJvbWlzZSByZXNvbHZpbmcgd2l0aCBleHBlcnQgYWR2aXNvciBmb3VuZFxuICAgKi9cbiAgZ2V0RXhwZXJ0QWR2aXNvcihhY2NvdW50SWQsIGV4cGVydElkKSB7XG4gICAgaWYgKHRoaXMuX2lzTm90Snd0VG9rZW4oKSkge1xuICAgICAgcmV0dXJuIHRoaXMuX2hhbmRsZU5vQWNjZXNzRXJyb3IoJ2dldEV4cGVydEFkdmlzb3JzJyk7XG4gICAgfVxuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvYWNjb3VudHMvJHthY2NvdW50SWR9L2V4cGVydC1hZHZpc29ycy8ke2V4cGVydElkfWAsXG4gICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVkIGV4cGVydCBhZHZpc29yIGRhdGFcbiAgICogQHR5cGVkZWYge09iamVjdH0gTmV3RXhwZXJ0QWR2aXNvckR0b1xuICAgKiBAcHJvcGVydHkge1N0cmluZ30gcGVyaW9kIGV4cGVydCBhZHZpc29yIHN5bWJvbC5cbiAgICogRm9yIE1ldGFUcmFkZXIgNCBhbGxvd2VkIHBlcmlvZHMgYXJlIDFtLCA1bSwgMTVtLCAzMG0sIDFoLCA0aCwgMWQsIDF3LCAxbW5cbiAgICogRm9yIE1ldGFUcmFkZXIgNSBhbGxvd2VkIHBlcmlvZHMgYXJlIDFtLCAybSwgM20sIDRtLCA1bSwgNm0sIDEwbSwgMTJtLCAxNW0sIDIwbSwgMzBtLCAxaCwgMmgsIDNoLCA0aCwgNmgsIDhoLCAxMmgsXG4gICAqIDFkLCAxdywgMW1uXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBzeW1ib2wgZXhwZXJ0IGFkdmlzb3IgcGVyaW9kXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBwcmVzZXQgYmFzZTY0LWVuY29kZWQgcHJlc2V0IGZpbGVcbiAgICovXG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgb3IgY3JlYXRlcyBleHBlcnQgYWR2aXNvciBkYXRhIChzZWVcbiAgICogaHR0cHM6Ly9tZXRhYXBpLmNsb3VkL2RvY3MvcHJvdmlzaW9uaW5nL2FwaS9leHBlcnRBZHZpc29yL3VwZGF0ZUV4cGVydEFkdmlzb3IvKS5cbiAgICogTWV0aG9kIGlzIGFjY2Vzc2libGUgb25seSB3aXRoIEFQSSBhY2Nlc3MgdG9rZW5cbiAgICogQHBhcmFtIHtTdHJpbmd9IGFjY291bnRJZCBNZXRhdHJhZGVyIGFjY291bnQgaWRcbiAgICogQHBhcmFtIHtTdHJpbmd9IGV4cGVydElkIGV4cGVydCBpZFxuICAgKiBAcGFyYW0ge05ld0V4cGVydEFkdmlzb3JEdG99IGV4cGVydCBuZXcgZXhwZXJ0IGFkdmlzb3IgZGF0YVxuICAgKiBAcmV0dXJucyB7UHJvbWlzZX0gcHJvbWlzZSByZXNvbHZpbmcgd2hlbiBleHBlcnQgYWR2aXNvciBpcyB1cGRhdGVkXG4gICAqL1xuICB1cGRhdGVFeHBlcnRBZHZpc29yKGFjY291bnRJZCwgZXhwZXJ0SWQsIGV4cGVydCkge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCdnZXRFeHBlcnRBZHZpc29ycycpO1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2FjY291bnRzLyR7YWNjb3VudElkfS9leHBlcnQtYWR2aXNvcnMvJHtleHBlcnRJZH1gLFxuICAgICAgbWV0aG9kOiAnUFVUJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIGpzb246IHRydWUsXG4gICAgICBib2R5OiBleHBlcnRcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxuICAvKipcbiAgICogVXBsb2FkcyBhbiBleHBlcnQgYWR2aXNvciBmaWxlIChzZWUgaHR0cHM6Ly9tZXRhYXBpLmNsb3VkL2RvY3MvcHJvdmlzaW9uaW5nL2FwaS9leHBlcnRBZHZpc29yL3VwbG9hZEVBRmlsZS8pXG4gICAqIE1ldGhvZCBpcyBhY2Nlc3NpYmxlIG9ubHkgd2l0aCBBUEkgYWNjZXNzIHRva2VuXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBhY2NvdW50SWQgTWV0YXRyYWRlciBhY2NvdW50IGlkXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBleHBlcnRJZCBleHBlcnQgaWRcbiAgICogQHBhcmFtIHtTdHJpbmd8QnVmZmVyfSBmaWxlIHBhdGggdG8gYSBmaWxlIHRvIHVwbG9hZCBvciBidWZmZXIgY29udGFpbmluZyBmaWxlIGNvbnRlbnRzXG4gICAqIEByZXR1cm5zIHtQcm9taXNlfSBwcm9taXNlIHJlc29sdmluZyB3aGVuIGZpbGUgdXBsb2FkIGlzIGNvbXBsZXRlZFxuICAgKi9cbiAgdXBsb2FkRXhwZXJ0QWR2aXNvckZpbGUoYWNjb3VudElkLCBleHBlcnRJZCwgZmlsZSkge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCd1cGxvYWRQcm92aXNpb25pbmdQcm9maWxlRmlsZScpO1xuICAgIH1cbiAgICBpZiAodHlwZW9mIGZpbGUgPT09ICdzdHJpbmcnKSB7XG4gICAgICBmaWxlID0gZnMuY3JlYXRlUmVhZFN0cmVhbShmaWxlKTtcbiAgICB9XG4gICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgIG1ldGhvZDogJ1BVVCcsXG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvYWNjb3VudHMvJHthY2NvdW50SWR9L2V4cGVydC1hZHZpc29ycy8ke2V4cGVydElkfS9maWxlYCxcbiAgICAgIGZvcm1EYXRhOiB7XG4gICAgICAgIGZpbGVcbiAgICAgIH0sXG4gICAgICBqc29uOiB0cnVlLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9XG4gICAgfTtcbiAgICByZXR1cm4gdGhpcy5faHR0cENsaWVudC5yZXF1ZXN0KG9wdHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gZXhwZXJ0IGFkdmlzb3IgKHNlZSBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9wcm92aXNpb25pbmcvYXBpL2V4cGVydEFkdmlzb3IvZGVsZXRlRXhwZXJ0QWR2aXNvci8pXG4gICAqIE1ldGhvZCBpcyBhY2Nlc3NpYmxlIG9ubHkgd2l0aCBBUEkgYWNjZXNzIHRva2VuXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBhY2NvdW50SWQgTWV0YXRyYWRlciBhY2NvdW50IGlkXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBleHBlcnRJZCBleHBlcnQgaWRcbiAgICogQHJldHVybnMge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gZXhwZXJ0IGFkdmlzb3IgaXMgZGVsZXRlZFxuICAgKi9cbiAgZGVsZXRlRXhwZXJ0QWR2aXNvcihhY2NvdW50SWQsIGV4cGVydElkKSB7XG4gICAgaWYgKHRoaXMuX2lzTm90Snd0VG9rZW4oKSkge1xuICAgICAgcmV0dXJuIHRoaXMuX2hhbmRsZU5vQWNjZXNzRXJyb3IoJ2dldEV4cGVydEFkdmlzb3JzJyk7XG4gICAgfVxuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvYWNjb3VudHMvJHthY2NvdW50SWR9L2V4cGVydC1hZHZpc29ycy8ke2V4cGVydElkfWAsXG4gICAgICBtZXRob2Q6ICdERUxFVEUnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG59XG4iXX0=