'use strict';

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _httpClient = require('./httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _errorHandler = require('./errorHandler');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {HttpClient#request}
 */
describe('HttpClient#request', () => {

  let httpClient;

  /**
   * @test {HttpClient#request}
   */
  describe('Real request', () => {

    beforeEach(() => {
      httpClient = new _httpClient2.default();
    });

    /**
     * @test {HttpClient#request}
     */
    it('should load HTML page from example.com', async () => {
      const opts = { url: 'http://example.com' };
      const response = await httpClient.request(opts);
      response.should.match(/doctype html/);
    });

    /**
     * @test {HttpClient#request}
     */
    it('should return NotFound error if server returns 404', async () => {
      let opts = { url: 'http://example.com/not-found' };
      try {
        const response = await httpClient.request(opts);
        _should2.default.not.exist(response);
      } catch (err) {
        err.name.should.be.eql('NotFoundError');
      }
    });

    /**
     * @test {HttpClient#request}
     */
    it('should return timeout error if request is timed out', async () => {
      httpClient = new _httpClient2.default(0.001, { retries: 2 });
      let opts = { url: 'http://metaapi.cloud' };
      try {
        const response = await httpClient.request(opts);
        _should2.default.not.exist(response);
      } catch (err) {
        err.name.should.be.eql('ApiError');
        err.message.should.be.equalOneOf('ETIMEDOUT', 'ESOCKETTIMEDOUT');
      }
    }).timeout(10000);
  });

  /**
   * @test {HttpClient#request}
   */
  describe('Retry request', () => {

    const opts = { url: 'http://metaapi.cloud' };
    let sandbox, stub, clock;

    before(() => {
      sandbox = _sinon2.default.createSandbox();
    });

    beforeEach(() => {
      stub = sandbox.stub();
      httpClient = new _httpClient.HttpClientMock(stub);
      clock = sandbox.useFakeTimers({
        shouldAdvanceTime: true,
        now: new Date('2020-10-05T07:00:00.000Z')
      });
    });

    afterEach(() => {
      sandbox.restore();
      clock.restore();
    });

    /**
     * @test {HttpClient#request}
     */
    describe('when InternalError or ApiError error occured', () => {

      /**
       * @test {HttpClient#request}
       */
      it('should retry request on fail with ApiError error', async () => {
        stub.onFirstCall().rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test')).onSecondCall().rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test')).onThirdCall().resolves('response');
        clock.tickAsync(5000);
        const response = await httpClient.request(opts);
        response.should.match('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should retry request on fail with InternalError error', async () => {
        stub.onFirstCall().rejects(new _errorHandler.InternalError('test')).onSecondCall().rejects(new _errorHandler.InternalError('test')).onThirdCall().resolves('response');
        clock.tickAsync(5000);
        const response = await httpClient.request(opts);
        response.should.match('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return error if retry limit exceeded', async () => {
        stub.rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test'));
        httpClient = new _httpClient.HttpClientMock(stub, 60, { retries: 2 });
        try {
          clock.tickAsync(5000);
          const response = await httpClient.request(opts);
          _should2.default.not.exist(response);
        } catch (err) {
          err.name.should.eql('ApiError');
          err.message.should.eql('test');
        }
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should not retry if error is neither InternalError nor ApiError', async () => {
        stub.onFirstCall().rejects(new _errorHandler.ValidationError('test')).onSecondCall().rejects(new _errorHandler.ValidationError('test')).onThirdCall().resolves('response');
        try {
          clock.tickAsync(5000);
          const response = await httpClient.request(opts);
          _should2.default.not.exist(response);
        } catch (err) {
          err.name.should.eql('ValidationError');
          err.message.should.eql('test');
        }
        _sinon2.default.assert.calledOnce(stub);
      }).timeout(10000);
    });

    /**
     * @test {HttpClient#request}
     */
    describe('when TooManyRequestsError error occured', () => {

      const getTooManyRequestsError = sec => {
        const date = new Date();
        date.setSeconds(date.getSeconds() + sec);
        const recommendedRetryTime = date.toUTCString();
        return new _errorHandler.TooManyRequestsError('test', { recommendedRetryTime });
      };

      /**
       * @test {HttpClient#request}
       */
      it('should retry request after waiting on fail with TooManyRequestsError error', async () => {
        stub.onFirstCall().rejects(getTooManyRequestsError(2)).onSecondCall().rejects(getTooManyRequestsError(3)).onThirdCall().resolves('response');
        clock.tickAsync(5000);
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return error if recommended retry time is too long', async () => {
        stub.onFirstCall().rejects(getTooManyRequestsError(2)).onSecondCall().rejects(getTooManyRequestsError(300)).onThirdCall().resolves('response');
        try {
          clock.tickAsync(5000);
          const response = await httpClient.request(opts);
          _should2.default.not.exist(response);
        } catch (err) {
          err.name.should.eql('TooManyRequestsError');
          err.message.should.eql('test');
        }
        _sinon2.default.assert.calledTwice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should not count retrying TooManyRequestsError error', async () => {
        stub.onFirstCall().rejects(getTooManyRequestsError(1)).onSecondCall().rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test')).onThirdCall().resolves('response');
        httpClient = new _httpClient.HttpClientMock(stub, 60, { retries: 1 });
        clock.tickAsync(5000);
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);
    });

    /**
     * @test {HttpClient#request}
     */
    describe('when status 202 response received', () => {

      /**
       * @test {HttpClient#request}
       */
      it('should wait for the retry-after header time before retrying', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 3 }, statusCode: 202 });
        }).onThirdCall().resolves('response');
        clock.tickAsync(5000);
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should wait for the retry-after header time before retrying if header is in http time', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 'Mon, 05 Oct 2020 07:00:02 GMT' },
            statusCode: 202 });
        }).onThirdCall().resolves('response');
        clock.tickAsync(3000);
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return TimeoutError error if retry-after header time is too long', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 30 }, statusCode: 202 });
        });
        httpClient = new _httpClient.HttpClientMock(stub, 60, { maxDelayInSeconds: 3 });
        try {
          await httpClient.request(opts);
          _should2.default.not.exist('Should not exist this assertion');
        } catch (err) {
          err.name.should.eql('TimeoutError');
          err.message.should.eql('Timed out waiting for the response');
        }
        _sinon2.default.assert.calledOnce(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return TimeoutError error if timed out to retry', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 1 }, statusCode: 202 });
        });
        httpClient = new _httpClient.HttpClientMock(stub, 60, { maxDelayInSeconds: 2, retries: 3 });
        try {
          clock.tickAsync(5000);
          await httpClient.request(opts);
          _should2.default.not.exist('Should not exist this assertion');
        } catch (err) {
          err.name.should.eql('TimeoutError');
          err.message.should.eql('Timed out waiting for the response');
        }
        _sinon2.default.assert.callCount(stub, 6);
      }).timeout(10000);
    });
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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