'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HttpClientMock = undefined;

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _requestPromiseAny = require('request-promise-any');

var _requestPromiseAny2 = _interopRequireDefault(_requestPromiseAny);

var _errorHandler = require('./errorHandler');

var _optionsValidator = require('./optionsValidator');

var _optionsValidator2 = _interopRequireDefault(_optionsValidator);

var _timeoutError = require('./timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * HTTP client library based on request-promise
 */
class HttpClient {

  /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   * @property {Number} [subscribeCooldownInSeconds] time to disable new subscriptions for
   */

  /**
   * Constructs HttpClient class instance
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */
  constructor(timeout = 60, retryOpts = {}) {
    const validator = new _optionsValidator2.default();
    this._timeout = timeout * 1000;
    this._retries = validator.validateNumber(retryOpts.retries, 5, 'retryOpts.retries');
    this._minRetryDelay = validator.validateNonZero(retryOpts.minDelayInSeconds, 1, 'retryOpts.minDelayInSeconds') * 1000;
    this._maxRetryDelay = validator.validateNonZero(retryOpts.maxDelayInSeconds, 30, 'retryOpts.maxDelayInSeconds') * 1000;
    this._logger = _logger2.default.getLogger('HttpClient');
  }

  /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */
  async request(options, retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries) {
    options.timeout = this._timeout;
    let retryAfterSeconds = 0;
    options.callback = (e, res) => {
      if (res && res.statusCode === 202) {
        retryAfterSeconds = res.headers['retry-after'];
        if (isNaN(retryAfterSeconds)) {
          retryAfterSeconds = Math.max((new Date(retryAfterSeconds).getTime() - Date.now()) / 1000, 1);
        }
      }
    };
    let body;
    try {
      body = await this._makeRequest(options);
    } catch (err) {
      retryCounter = await this._handleError(err, retryCounter, endTime);
      return this.request(options, retryCounter, endTime);
    }
    if (retryAfterSeconds) {
      if (body && body.message) {
        this._logger.info(`Retrying request in ${Math.floor(retryAfterSeconds)} seconds because request ` + 'returned message:', body.message);
      }
      await this._handleRetry(endTime, retryAfterSeconds * 1000);
      body = await this.request(options, retryCounter, endTime);
    }
    return body;
  }

  _makeRequest(options) {
    return (0, _requestPromiseAny2.default)(options);
  }

  async _wait(pause) {
    await new _promise2.default(res => setTimeout(res, pause));
  }

  async _handleRetry(endTime, retryAfter) {
    if (endTime > Date.now() + retryAfter) {
      await this._wait(retryAfter);
    } else {
      throw new _timeoutError2.default('Timed out waiting for the response');
    }
  }

  async _handleError(err, retryCounter, endTime) {
    const error = this._convertError(err);
    if (['ConflictError', 'InternalError', 'ApiError', 'TimeoutError'].includes(error.name) && retryCounter < this._retries) {
      const pause = Math.min(Math.pow(2, retryCounter) * this._minRetryDelay, this._maxRetryDelay);
      await this._wait(pause);
      return retryCounter + 1;
    } else if (error.name === 'TooManyRequestsError') {
      const retryTime = Date.parse(error.metadata.recommendedRetryTime);
      if (retryTime < endTime) {
        await this._wait(retryTime - Date.now());
        return retryCounter;
      }
    }
    throw error;
  }

  // eslint-disable-next-line complexity
  _convertError(err) {
    err.error = err.error || {};
    let status = err.statusCode || err.status;
    switch (status) {
      case 400:
        return new _errorHandler.ValidationError(err.error.message || err.message, err.error.details || err.details);
      case 401:
        return new _errorHandler.UnauthorizedError(err.error.message || err.message);
      case 403:
        return new _errorHandler.ForbiddenError(err.error.message || err.message);
      case 404:
        return new _errorHandler.NotFoundError(err.error.message || err.message);
      case 409:
        return new _errorHandler.ConflictError(err.error.message || err.message);
      case 429:
        return new _errorHandler.TooManyRequestsError(err.error.message || err.message, err.error.metadata || err.metadata);
      case 500:
        return new _errorHandler.InternalError(err.error.message || err.message);
      default:
        return new _errorHandler.ApiError(_errorHandler.ApiError, err.error.message || err.message, status);
    }
  }

}

exports.default = HttpClient; /**
                               * HTTP client service mock for tests
                               */

class HttpClientMock extends HttpClient {

  /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */
  constructor(requestFn, timeout, retryOpts) {
    super(timeout, retryOpts);
    this._requestFn = requestFn;
  }

  _makeRequest() {
    return this._requestFn.apply(this, arguments);
  }

}
exports.HttpClientMock = HttpClientMock;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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