'use strict';

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _trading = require('./trading.client');

var _trading2 = _interopRequireDefault(_trading);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const copyFactoryApiUrl = 'https://copyfactory-application-history-master-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {TradingClient}
 */
describe('TradingClient', () => {

  let tradingClient;
  const token = 'header.payload.sign';
  let httpClient = new _httpClient2.default();
  let sandbox;
  let requestStub;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    tradingClient = new _trading2.default(httpClient, token);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {TradingClient#generateSignalId}
   */
  it('should generate signal id', async () => {
    tradingClient.generateSignalId().length.should.equal(8);
  });

  /**
   * @test {TradingClient#updateExternalSignal}
   */
  it('should update external signal', async () => {
    const signal = {
      symbol: 'EURUSD',
      type: 'POSITION_TYPE_BUY',
      time: '2020-08-24T00:00:00.000Z',
      updateTime: '2020-08-24T00:00:00.000Z',
      volume: 1
    };
    await tradingClient.updateExternalSignal('ABCD', '0123456', signal);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/strategies/ABCD/external-signals/0123456`,
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      json: true,
      body: signal });
  });

  /**
   * @test {TradingClient#updateExternalSignal}
   */
  it('should not update external signal with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.updateExternalSignal('ABCD', '0123456', {});
    } catch (error) {
      error.message.should.equal('You can not invoke updateExternalSignal method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#removeExternalSignal}
   */
  it('should remove external signal', async () => {
    const signal = {
      time: '2020-08-24T00:00:00.000Z'
    };
    await tradingClient.removeExternalSignal('ABCD', '0123456', signal);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/strategies/ABCD/external-signals/0123456/remove`,
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true,
      body: signal });
  });

  /**
   * @test {TradingClient#removeExternalSignal}
   */
  it('should not remove external signal with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.removeExternalSignal('ABCD', '0123456', {});
    } catch (error) {
      error.message.should.equal('You can not invoke removeExternalSignal method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#resynchronize}
   */
  it('should resynchronize CopyFactory account', async () => {
    await tradingClient.resynchronize('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', ['ABCD'], ['0123456']);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/subscribers/e8867baa-5ec2-45ae-9930-4d5cea18d0d6/resynchronize`,
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true,
      qs: {
        strategyId: ['ABCD'],
        positionId: ['0123456']
      }
    });
  });

  /**
   * @test {TradingClient#resynchronize}
   */
  it('should not resynchronize account with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.resynchronize('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    } catch (error) {
      error.message.should.equal('You can not invoke resynchronize method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#getTradingSignals}
   */
  it('should retrieve signals', async () => {
    const expected = [{
      symbol: 'EURUSD',
      type: 'POSITION_TYPE_BUY',
      time: '2020-08-24T00:00:00.000Z',
      closeAfter: '2020-08-24T00:00:00.000Z',
      volume: 1
    }];
    requestStub.resolves(expected);
    let stopouts = await tradingClient.getTradingSignals('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    stopouts.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/subscribers/e8867baa-5ec2-45ae-9930-4d5cea18d0d6/signals`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {TradingClient#getTradingSignals}
   */
  it('should not retrieve signals with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.getTradingSignals('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    } catch (error) {
      error.message.should.equal('You can not invoke getTradingSignals method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#getStopouts}
   */
  it('should retrieve stopouts', async () => {
    let expected = [{
      reason: 'max-drawdown',
      stoppedAt: new Date('2020-08-08T07:57:30.328Z'),
      strategy: {
        id: 'ABCD',
        name: 'Strategy'
      },
      reasonDescription: 'total strategy equity drawdown exceeded limit'
    }];
    requestStub.resolves(expected);
    let stopouts = await tradingClient.getStopouts('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    stopouts.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/subscribers/e8867baa-5ec2-45ae-9930-4d5cea18d0d6/stopouts`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {TradingClient#getStopouts}
   */
  it('should not retrieve stopouts from API with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.getStopouts('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    } catch (error) {
      error.message.should.equal('You can not invoke getStopouts method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#resetStopouts}
   */
  it('should reset stopouts', async () => {
    await tradingClient.resetStopouts('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', 'ABCD', 'daily-equity');
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/subscribers/` + 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6/subscription-strategies/ABCD/stopouts/daily-equity/reset',
      method: 'POST',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {TradingClient#resetStopouts}
   */
  it('should not reset stopouts with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.resetStopouts('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', 'ABCD', 'daily-equity');
    } catch (error) {
      error.message.should.equal('You can not invoke resetStopouts method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#getUserLog}
   */
  it('should retrieve copy trading user log', async () => {
    let expected = [{
      time: new Date('2020-08-08T07:57:30.328Z'),
      level: 'INFO',
      message: 'message'
    }];
    requestStub.resolves(expected);
    let records = await tradingClient.getUserLog('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', new Date('2020-08-01T00:00:00.000Z'), new Date('2020-08-10T00:00:00.000Z'), 10, 100);
    records.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/subscribers/e8867baa-5ec2-45ae-9930-4d5cea18d0d6/user-log`,
      method: 'GET',
      qs: {
        startTime: new Date('2020-08-01T00:00:00.000Z'),
        endTime: new Date('2020-08-10T00:00:00.000Z'),
        offset: 10,
        limit: 100
      },
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {TradingClient#getUserLog}
   */
  it('should not retrieve copy trading user log from API with account token', async () => {
    tradingClient = new _trading2.default(httpClient, 'token');
    try {
      await tradingClient.getUserLog('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    } catch (error) {
      error.message.should.equal('You can not invoke getUserLog method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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