'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

var _randomstring = require('randomstring');

var _randomstring2 = _interopRequireDefault(_randomstring);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud CopyFactory trading API (trade copying trading API) client (see
 * https://metaapi.cloud/docs/copyfactory/)
 */
class TradingClient extends _metaApi2.default {

  /**
   * Constructs CopyFactory trading API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */
  constructor(httpClient, token, domain = 'agiliumtrade.agiliumtrade.ai') {
    super(httpClient, token, domain);
    this._host = `https://copyfactory-application-history-master-v1.${domain}`;
  }

  /**
   * Generates random signal id
   * @return {String} signal id
   */
  generateSignalId() {
    return _randomstring2.default.generate(8);
  }
  /**
   * CopyFactory external signal update payload
   * @typedef {Object} CopyFactoryExternalSignalUpdate
   * @property {String} symbol trade symbol
   * @property {String} type trade type (one of POSITION_TYPE_BUY, POSITION_TYPE_SELL, ORDER_TYPE_BUY_LIMIT, ORDER_TYPE_SELL_LIMIT, 
   * ORDER_TYPE_BUY_STOP, ORDER_TYPE_SELL_STOP)
   * @property {Date} time time the signal was emitted at
   * @property {Date} [updateTime] last time of the signal update
   * @property {Number} volume volume traded
   * @property {Number} [magic] expert advisor id
   * @property {Number} [stopLoss] stop loss price
   * @property {Number} [takeProfit] take profit price
   * @property {Number} [openPrice] pending order open price
   */

  /**
   * CopyFactory trading signal
   * @typedef {Object} CopyFactoryTradingSignal
   * @property {CopyFactoryStrategyIdAndName} strategy strategy the signal arrived from
   * @property {String} positionId id of the position the signal was generated from
   * @property {Date} time signal time
   * @property {String} symbol symbol traded
   * @property {String} type type of the trade (one of market, limit, stop)
   * @property {String} side side of the trade (one of buy, sell, close)
   * @property {Number} [openPrice] open price for limit and stop orders
   * @property {Number} [stopLoss] stop loss price
   * @property {Number} [takeProfit] take profit price
   * @property {Number} signalVolume the signal volume
   * @property {Number} subscriberVolume the volume already open on subscriber side
   * @property {Number} subscriberProfit total profit of the position on subscriber side
   * @property {Date} closeAfter the time the signal will be automatically closed at
   * @property {Boolean} [closeOnly] flag indicating that only closing side of this signal will be copied
   */

  /**
   * Updates external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/updateExternalSignal/
   * @param {String} strategyId strategy id
   * @param {String} signalId external signal id (should be 8 alphanumerical symbols)
   * @param {CopyFactoryExternalSignalUpdate} signal signal update payload
   * @return {Promise} promise which resolves when the external signal is updated
   */
  updateExternalSignal(strategyId, signalId, signal) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('updateExternalSignal');
    }
    const opts = {
      url: `${this._host}/users/current/strategies/${strategyId}/external-signals/${signalId}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      body: signal,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * CopyFactory external signal remove payload
   * @typedef {Object} CopyFactoryExternalSignalRemove
   * @property {Date} time the time signal was removed (closed) at
   */

  /**
   * Updates external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/removeExternalSignal/
   * @param {String} strategyId strategy id
   * @param {String} signalId external signal id
   * @param {CopyFactoryExternalSignalRemove} signal signal removal payload
   * @return {Promise} promise which resolves when the external signal is removed
   */
  removeExternalSignal(strategyId, signalId, signal) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('removeExternalSignal');
    }
    const opts = {
      url: `${this._host}/users/current/strategies/${strategyId}/external-signals/${signalId}/remove`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      body: signal,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Resynchronizes the account. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/resynchronize/
   * @param {String} accountId account id
   * @param {Array<String>} [strategyIds] array of strategy ids to recynchronize. Default is to synchronize all
   * strategies
   * @param {Array<String>} [positionIds] array of position ids to resynchronize. Default is to synchronize all
   * positions
   * @return {Promise} promise which resolves when resynchronization is scheduled
   */
  async resynchronize(accountId, strategyIds, positionIds) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('resynchronize');
    }
    const opts = {
      url: `${this._host}/users/current/subscribers/${accountId}/resynchronize`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      qs: {
        strategyId: strategyIds,
        positionId: positionIds
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Returns trading signals the subscriber is subscribed to. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getTradingSignals/
   * @param {String} subscriberId subscriber id
   * @returns {Promise<Array<CopyFactoryTradingSignal>>}
   */
  getTradingSignals(subscriberId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getTradingSignals');
    }
    const opts = {
      url: `${this._host}/users/current/subscribers/${subscriberId}/signals`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * CopyFactory strategy stopout
   * @typedef {Object} CopyFactoryStrategyStopout
   * @property {CopyFactoryStrategyIdAndName} strategy strategy which was stopped out
   * @property {Boolean} partial flag indicating that stopout is partial
   * @property {String} reason stopout reason. One of yearly-balance, monthly-balance, daily-balance, yearly-equity,
   * monthly-equity, daily-equity, max-drawdown
   * @property {String} reasonDescription human-readable description of the stopout reason
   * @property {Boolean} [closePositions] flag indicating if positions should be closed
   * @property {Date} stoppedAt time the strategy was stopped at
   * @property {Date} stoppedTill time the strategy is stopped till
   */

  /**
   * Returns subscriber account stopouts. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getStopOuts/
   * @param {String} subscriberId subscriber id
   * @return {Promise<Array<CopyFactoryStrategyStopout>>} promise which resolves with stopouts found
   */
  async getStopouts(subscriberId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getStopouts');
    }
    const opts = {
      url: `${this._host}/users/current/subscribers/${subscriberId}/stopouts`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Resets strategy stopouts. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/resetStopOuts/
   * @param {String} subscriberId subscriber id
   * @param {String} strategyId strategy id
   * @param {String} reason stopout reason to reset. One of yearly-balance, monthly-balance, daily-balance,
   * yearly-equity, monthly-equity, daily-equity, max-drawdown
   * @return {Promise} promise which resolves when the stopouts are reset
   */
  async resetStopouts(subscriberId, strategyId, reason) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('resetStopouts');
    }
    const opts = {
      url: `${this._host}/users/current/subscribers/${subscriberId}/subscription-strategies/` + `${strategyId}/stopouts/${reason}/reset`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Trade copying user log record
   * @typedef {Object} CopyFactoryUserLogMessage
   * @property {Date} time log record time
   * @property {string} [symbol] symbol traded
   * @property {string} [strategyId] id of the strategy event relates to
   * @property {string} [strategyName] name of the strategy event relates to
   * @property {string} [positionId] position id event relates to
   * @property {string} [side] side of the trade event relates to. One of buy, sell, close
   * @property {string} [type] type of the trade event relates to. One of market, limit, stop
   * @property {number} [openPrice] open price for limit and stop orders
   * @property {string} level log level. One of INFO, WARN, ERROR
   * @property {string} message log message
   */

  /**
   * Returns copy trading user log for an account and time range. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getUserLog/
   * @param {string} subscriberId subscriber id
   * @param {Date} [startTime] time to start loading data from
   * @param {Date} [endTime] time to stop loading data at
   * @param {number} [offset] pagination offset. Default is 0
   * @param {number} [limit] pagination limit. Default is 1000
   * @return {Promise<Array<CopyFactoryUserLogMessage>>} promise which resolves with log records found
   */
  async getUserLog(subscriberId, startTime, endTime, offset = 0, limit = 1000) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getUserLog');
    }
    const opts = {
      url: `${this._host}/users/current/subscribers/${subscriberId}/user-log`,
      method: 'GET',
      qs: {
        startTime,
        endTime,
        offset,
        limit
      },
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    let result = await this._httpClient.request(opts);
    if (result) {
      result.map(r => r.time = new Date(r.time));
    }
    return result;
  }

}
exports.default = TradingClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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