'use strict';

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _history = require('./history.client');

var _history2 = _interopRequireDefault(_history);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const copyFactoryApiUrl = 'https://copyfactory-application-history-master-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {TradingClient}
 */
describe('HistoryClient', () => {

  let copyFactoryClient;
  const token = 'header.payload.sign';
  let httpClient = new _httpClient2.default();
  let sandbox;
  let requestStub;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    copyFactoryClient = new _history2.default(httpClient, token);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {TradingClient#getProvidedTransactions}
   */
  it('should retrieve transactions performed on provided strategies from API', async () => {
    let expected = [{
      id: '64664661:close',
      type: 'DEAL_TYPE_SELL',
      time: new Date('2020-08-02T21:01:01.830Z'),
      subscriberId: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      symbol: 'EURJPY',
      subscriberUser: {
        id: 'subscriberId',
        name: 'Subscriber'
      },
      demo: false,
      providerUser: {
        id: 'providerId',
        name: 'Provider'
      },
      strategy: {
        id: 'ABCD'
      },
      improvement: 0,
      providerCommission: 0,
      platformCommission: 0,
      quantity: -0.04,
      lotPrice: 117566.08744776,
      tickPrice: 124.526,
      amount: -4702.643497910401,
      commission: -0.14,
      swap: -0.14,
      profit: 0.49
    }];
    let from = new Date();
    let till = new Date();
    requestStub.resolves(expected);
    let transactions = await copyFactoryClient.getProvidedTransactions(from, till, ['ABCD'], ['e8867baa-5ec2-45ae-9930-4d5cea18d0d6'], 100, 200);
    transactions.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/provided-transactions`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      qs: {
        from,
        till,
        strategyId: ['ABCD'],
        subscriberId: ['e8867baa-5ec2-45ae-9930-4d5cea18d0d6'],
        offset: 100,
        limit: 200
      },
      json: true
    });
  });

  /**
   * @test {TradingClient#getProvidedTransactions}
   */
  it('should not retrieve transactions on provided strategies from API with account token', async () => {
    copyFactoryClient = new _history2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getProvidedTransactions(new Date(), new Date());
    } catch (error) {
      error.message.should.equal('You can not invoke getProvidedTransactions method, because you have connected with account ' + 'access token. Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#getSubscriptionTransactions}
   */
  it('should retrieve transactions performed on strategies current user is subscribed to from API', async () => {
    let expected = [{
      id: '64664661:close',
      type: 'DEAL_TYPE_SELL',
      time: new Date('2020-08-02T21:01:01.830Z'),
      subscriberId: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      symbol: 'EURJPY',
      subscriberUser: {
        id: 'subscriberId',
        name: 'Subscriber'
      },
      demo: false,
      providerUser: {
        id: 'providerId',
        name: 'Provider'
      },
      strategy: {
        id: 'ABCD'
      },
      improvement: 0,
      providerCommission: 0,
      platformCommission: 0,
      quantity: -0.04,
      lotPrice: 117566.08744776,
      tickPrice: 124.526,
      amount: -4702.643497910401,
      commission: -0.14,
      swap: -0.14,
      profit: 0.49
    }];
    let from = new Date();
    let till = new Date();
    requestStub.resolves(expected);
    let transactions = await copyFactoryClient.getSubscriptionTransactions(from, till, ['ABCD'], ['e8867baa-5ec2-45ae-9930-4d5cea18d0d6'], 100, 200);
    transactions.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/subscription-transactions`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true,
      qs: {
        from,
        till,
        strategyId: ['ABCD'],
        subscriberId: ['e8867baa-5ec2-45ae-9930-4d5cea18d0d6'],
        offset: 100,
        limit: 200
      }
    });
  });

  /**
   * @test {TradingClient#getSubscriptionTransactions}
   */
  it('should not retrieve transactions on strategies subscribed to from API with account token', async () => {
    copyFactoryClient = new _history2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getSubscriptionTransactions(new Date(), new Date());
    } catch (error) {
      error.message.should.equal('You can not invoke getSubscriptionTransactions method, because you have connected with account ' + 'access token. Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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